<?php
/**
 * Custom Dimensions Tracking Class
 *
 * Handles tracking of custom dimensions for Google Analytics (UA and GA4).
 * Supports various dimension types including post type, author, category, tags,
 * published date, SEO score, focus keyword, and archive type.
 *
 * @package WP_Analytify_Pro
 * @since 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class Analytify_Dimensions_Tracking
 *
 * @since 1.0.0
 */
class Analytify_Dimensions_Tracking {

	/**
	 * Google Analytics measurement ID.
	 *
	 * @var string
	 */
	protected $measurement_id;

	/**
	 * Class constructor.
	 *
	 * Initializes the measurement ID and sets up hooks for tracking.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		if ( method_exists( 'WP_ANALYTIFY_FUNCTIONS', 'get_UA_code' ) ) {
			$this->measurement_id = WP_ANALYTIFY_FUNCTIONS::get_UA_code();
		}
		$this->init_hooks();
	}

	/**
	 * Initialize WordPress hooks for dimension tracking.
	 *
	 * Sets up different tracking methods based on GA mode (GA4 vs UA) and tracking mode (gtag vs analytics.js).
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public function init_hooks() {
		// For Google Analytics V4 with gtag.
		if ( method_exists( 'WPANALYTIFY_Utils', 'get_ga_mode' )
			&& 'ga4' === WPANALYTIFY_Utils::get_ga_mode()
			&& 'gtag' === ANALYTIFY_TRACKING_MODE ) {
			add_filter( 'analytify_gtag_configuration', array( $this, 'wpa_add_dimension_configuration' ) );
			add_action( 'analytify_tracking_code_after_pageview', array( $this, 'wpa_emit_page_view_event_params' ) );
			return;
		}

		// For Universal Analytics or analytics.js mode.
		add_action( 'analytify_tracking_code_before_pageview', array( $this, 'wpa_add_dimension_hit_code' ) );
	}

	/**
	 * Add dimensions data to gtag configuration for GA4.
	 *
	 * Filters the gtag configuration array to include custom dimension values.
	 * Prevents duplicate page_view events and sets up user_id if available.
	 *
	 * @since 1.0.0
	 * @version 7.0.4
	 * @param array $configuration The gtag configuration array.
	 * @return array Modified configuration array with dimension values.
	 */
	public function wpa_add_dimension_configuration( $configuration ) {
		$current_dimensions = $GLOBALS['WP_ANALYTIFY']->settings->get_option( 'analytiy_custom_dimensions', 'wp-analytify-custom-dimensions' );
		$defined_dimensions = Analytify_Google_Dimensions::get_current_dimensions();

		// Check if user has added dimensions.
		if ( empty( $current_dimensions ) || ! is_array( $current_dimensions ) ) {
			return $configuration;
		}

		// Check if we actually have dimension data to send on this page
		$has_dimension_data = false;
		$dimension_params = array();

		// Bind GA4-recognized user_id if present.
		$uid = $this->wpa_track_user_id();
		if ( ! empty( $uid ) ) {
			$dimension_params['user_id'] = $uid;
		}

		foreach ( $current_dimensions as $key => $value ) {
			$type       = isset( $value['type'] ) ? $value['type'] : '';
			$identifier = isset( $defined_dimensions[ $type ]['value'] ) ? $defined_dimensions[ $type ]['value'] : false;

			if ( ! $identifier ) {
				continue;
			}

			$tracking_val = $this->wpa_get_tracking_value( $type );

			// For logged_in, 'false' is a valid value (user not logged in)
			// Check if value exists (not empty string) OR if it's the string 'false'
			if ( ! empty( $tracking_val ) || ( 'logged_in' === $type && 'false' === $tracking_val ) ) {
				$has_dimension_data = true;
				$configuration[ $identifier ] = $tracking_val;
			}
		}

		// Only disable automatic page_view if we have dimension data to send
		// This allows automatic page_view to work normally on pages without dimensions
		if ( $has_dimension_data ) {
			// Prevent duplicate page_view; we'll emit one manually with dimension params.
			$configuration['send_page_view'] = false;
		}

		// Keep DebugView visibility.
		$configuration['debug_mode'] = true;

		return $configuration;
	}

	/**
	 * Get tracking value for a specific dimension type.
	 *
	 * @since 1.0.0
	 * @version 7.0.4
	 * 
	 * @param string $type The dimension type (e.g., 'post_type', 'author', 'category').
	 * @return mixed The tracking value for the dimension type, empty string if not found.
	 */
	private function wpa_get_tracking_value( $type ) {
		switch ( $type ) {
			case 'logged_in':
				return $this->wpa_track_logged_in();

			case 'user_id':
				return $this->wpa_track_user_id();

			case 'post_type':
				return $this->wpa_track_post_type();

			case 'author':
				return $this->wpa_track_author();

			case 'category':
				return $this->wpa_track_category();

			case 'tags':
				return $this->wpa_track_tags();

			case 'published_at':
				return $this->wpa_track_published_at();

			case 'seo_score':
				return $this->wpa_track_seo_score();

			case 'focus_keyword':
				return $this->wpa_track_focus_keyword();

			case 'archive_type':
				return $this->wpa_track_archive_type();

			default:
				return '';
		}
	}

	/**
	 * Callback for dimensions hit code generator (Universal Analytics).
	 *
	 * Outputs dimension tracking code for Universal Analytics before pageview.
	 *
	 * @since 1.0.0
	 * @version 7.0.4
	 * @return void
	 */
	public function wpa_add_dimension_hit_code() {
		$this->wpa_track_dimension();
	}

	/**
	 * Track all selected dimensions for Universal Analytics.
	 *
	 * Outputs JavaScript code to set custom dimensions before pageview tracking.
	 * Works with both gtag and analytics.js tracking modes.
	 *
	 * @since 1.0.0
	 * @version 7.0.4
	 * @return void
	 */
	public function wpa_track_dimension() {
		$current_dimensions = $GLOBALS['WP_ANALYTIFY']->settings->get_option( 'analytiy_custom_dimensions', 'wp-analytify-custom-dimensions' );
		$defined_dimensions = Analytify_Google_Dimensions::get_current_dimensions();

		if ( empty( $current_dimensions ) || ! is_array( $current_dimensions ) ) {
			return;
		}

		foreach ( $current_dimensions as $key => $value ) {
			$type = isset( $value['type'] ) ? $value['type'] : '';
			$id   = isset( $value['id'] ) ? $value['id'] : ( isset( $defined_dimensions[ $type ]['id'] ) ? $defined_dimensions[ $type ]['id'] : '' );

			if ( empty( $id ) ) {
				continue;
			}

			$tracking_val = $this->wpa_get_tracking_value( $type );

			if ( ! empty( $tracking_val ) ) {
				$script = $this->wpa_get_dimension_script( $id, $tracking_val );
				if ( 'gtag' === ANALYTIFY_TRACKING_MODE ) {
					echo 'gtag(' . esc_js( $script ) . ');'; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
				} else {
					echo 'ga(' . esc_js( $script ) . ');'; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
				}
			}
		}
	}

	/**
	 * Generate dimensions hit script for Universal Analytics.
	 *
	 * Creates the JavaScript code string for setting a custom dimension.
	 *
	 * @since 1.0.0
	 * @version 7.0.4
	 * @param int|string $id          The dimension ID (1-200 for UA).
	 * @param string     $tracking_val The value to track for this dimension.
	 * @return string The formatted JavaScript code string.
	 */
	public function wpa_get_dimension_script( $id, $tracking_val ) {
		return "'set', 'dimension" . absint( $id ) . "', '" . esc_js( addslashes( $tracking_val ) ) . "'";
	}

	/**
	 * Check if user is logged in.
	 *
	 * Returns a string representation of the logged-in status ('true' or 'false').
	 *
	 * @since 1.0.0
	 * @version 7.0.4
	 * @return string 'true' if user is logged in, 'false' otherwise.
	 */
	public function wpa_track_logged_in() {
		return var_export( is_user_logged_in(), true );
	}

	/**
	 * Get current logged in user ID.
	 *
	 * Returns the current user ID if logged in, otherwise returns 0.
	 *
	 * @since 1.0.0
	 * @version 7.0.4
	 * @return int Current user ID, or 0 if not logged in.
	 */
	public function wpa_track_user_id() {
		return is_user_logged_in() ? get_current_user_id() : 0;
	}

	/**
	 * Get the post type of current post.
	 *
	 * Returns the post type slug for singular pages, empty string otherwise.
	 *
	 * @since 1.0.0
	 * @version 7.0.4
	 * @return string Post type slug, or empty string if not singular.
	 */
	public function wpa_track_post_type() {
		return is_singular() ? get_post_type( get_the_ID() ) : '';
	}

	/**
	 * Get the post author's display name.
	 *
	 * Returns the author's display name for singular posts.
	 * Uses display_name instead of user_login for security/privacy.
	 *
	 * @since 1.0.0
	 * @version 7.0.4
	 * @return string Author display name, or empty string if not singular or no author.
	 */
	public function wpa_track_author() {
		$value = '';

		if ( is_singular() ) {
			// Reset post data to ensure we have the correct post context.
			if ( have_posts() ) {
				while ( have_posts() ) {
					the_post();
				}
			}

			// SECURITY: Use display_name instead of user_login (less sensitive).
			$author_id = get_the_author_meta( 'ID' );
			if ( $author_id ) {
				$author_info = get_userdata( $author_id );
				if ( $author_info && isset( $author_info->display_name ) ) {
					$value = $author_info->display_name;
				}
			}
		}

		return $value;
	}

	/**
	 * Get the comma-separated categories of current post.
	 *
	 * Returns primary category from Yoast SEO if available, otherwise returns
	 * comma-separated list of category slugs.
	 *
	 * @since 1.0.0
	 * @version 7.0.4
	 * @return string Comma-separated category names/slugs, or empty string.
	 */
	public function wpa_track_category() {
		$value = '';

		if ( is_single() ) {
			// Check primary category from Yoast SEO.
			$main_category_id = get_post_meta( get_the_ID(), '_yoast_wpseo_primary_category', true );

			if ( ! empty( $main_category_id ) ) {
				$main_category = get_category( $main_category_id );
				if ( $main_category && isset( $main_category->name ) ) {
					$value = $main_category->name;
				}
			}

			// Fallback to all categories if no primary category.
			if ( empty( $value ) ) {
				$categories = get_the_category( get_the_ID() );

				if ( $categories && is_array( $categories ) ) {
					$category_names = array();
					foreach ( $categories as $category ) {
						if ( isset( $category->slug ) ) {
							$category_names[] = $category->slug;
						}
					}
					$value = implode( ',', $category_names );
				}
			}
		}

		return $value;
	}

	/**
	 * Get the comma-separated tags of current post.
	 *
	 * Returns a comma-separated list of tag names for singular posts.
	 *
	 * @since 1.0.0
	 * @version 7.0.4
	 * @return string Comma-separated tag names, or empty string.
	 */
	public function wpa_track_tags() {
		$tag_names = '';

		if ( is_single() ) {
			$tags = get_the_tags( get_the_ID() );
			if ( $tags && is_array( $tags ) ) {
				$tag_names = implode( ',', wp_list_pluck( $tags, 'name' ) );
			}
		}

		return $tag_names;
	}

	/**
	 * Get the publish date and time of current post.
	 *
	 * Returns the post publish date in ISO 8601 format (RFC 3339).
	 *
	 * @since 1.0.0
	 * @version 7.0.4
	 * @return string Post publish date in ISO 8601 format, or empty string.
	 */
	public function wpa_track_published_at() {
		return is_singular() ? get_the_date( 'c' ) : '';
	}

	/**
	 * Get Yoast SEO score of current post.
	 *
	 * Returns the Yoast SEO readability score (linkdex) for singular posts.
	 * Only works if Yoast SEO plugin is active.
	 *
	 * @since 1.0.0
	 * @version 7.0.4
	 * @return string|false SEO score value, or false if Yoast SEO not available or not singular.
	 */
	public function wpa_track_seo_score() {
		if ( class_exists( 'WPSEO_Frontend' ) && is_singular() ) {
			$score = WPSEO_Metabox::get_value( 'linkdex', get_the_ID() );
			return $score;
		}

		return false;
	}

	/**
	 * Get Yoast SEO focus keyword of current post.
	 *
	 * Returns the focus keyword meta value from Yoast SEO for singular posts.
	 *
	 * @since 1.0.0
	 * @version 7.0.4
	 * @return string Focus keyword, or empty string.
	 */
	public function wpa_track_focus_keyword() {
		$focus_keyword = '';

		if ( is_singular() ) {
			$focus_keyword = get_post_meta( get_the_ID(), '_yoast_wpseo_focuskw', true );
		}

		return $focus_keyword;
	}

	/**
	 * Get the archive type of current page.
	 *
	 * Determines and returns the type of archive page being viewed.
	 * Uses early returns for better performance and optimized check order.
	 * Supports: category, tag, author, date (year/month/day), taxonomy,
	 * post type archive, blog page, and search results.
	 *
	 * @since 7.0.4
	 * @return string Archive type identifier, or empty string if not an archive page.
	 */
	public function wpa_track_archive_type() {
		// Most common archive types checked first for performance.
		if ( is_category() ) {
			return 'category';
		}

		if ( is_tag() ) {
			return 'tag';
		}

		if ( is_author() ) {
			return 'author';
		}

		if ( is_search() ) {
			return 'search';
		}

		// Date archives with nested checks.
		if ( is_date() ) {
			if ( is_year() ) {
				return 'year';
			}
			if ( is_month() ) {
				return 'month';
			}
			if ( is_day() ) {
				return 'day';
			}
			return 'date';
		}

		// Custom taxonomy archive - use queried object for reliability.
		if ( is_tax() ) {
			$queried_object = get_queried_object();
			if ( $queried_object && isset( $queried_object->taxonomy ) ) {
				return 'taxonomy:' . sanitize_key( $queried_object->taxonomy );
			}
			return 'taxonomy';
		}

		// Custom post type archive - use queried object for better reliability.
		if ( is_post_type_archive() ) {
			$queried_object = get_queried_object();
			if ( $queried_object && isset( $queried_object->name ) ) {
				return 'post_type:' . sanitize_key( $queried_object->name );
			}
			// Fallback to query var if queried object not available.
			$post_type = get_query_var( 'post_type' );
			if ( is_array( $post_type ) ) {
				$post_type = reset( $post_type );
			}
			if ( $post_type ) {
				return 'post_type:' . sanitize_key( $post_type );
			}
			return 'post_type';
		}

		// Blog posts page (home but not front page).
		if ( is_home() && ! is_front_page() ) {
			return 'blog';
		}

		// Not an archive page.
		return '';
	}

	/**
	 * Emit page_view event with custom dimension parameters for GA4.
	 *
	 * Outputs JavaScript code to send a page_view event with all tracked
	 * custom dimensions as event parameters. Only emits if dimensions are present.
	 *
	 * @since 7.0.4
	 * @return void
	 */
	public function wpa_emit_page_view_event_params() {
		// Get configured custom dimensions.
		$current_dimensions = $GLOBALS['WP_ANALYTIFY']->settings->get_option( 'analytiy_custom_dimensions', 'wp-analytify-custom-dimensions' );
		if ( empty( $current_dimensions ) || ! is_array( $current_dimensions ) ) {
			return;
		}

		// Build GA4 event parameters.
		$params = $this->wpa_get_ga4_event_params();

		$uid = $this->wpa_track_user_id();
		if ( ! empty( $uid ) ) {
			$params['user_id'] = $uid;
		}

		// Only send manual page_view if we have dimension data
		// If no dimension data, let automatic page_view work normally
		// Note: Check count() instead of empty() because 'false' values are valid dimension data
		if ( count( $params ) === 0 ) {
			return;
		}

		// Set user_id if available.
		if ( ! empty( $uid ) ) {
			echo "\n\t\tgtag('set', {'user_id': " . wp_json_encode( $uid ) . "});";
		}
		// Send page_view event with custom dimension parameters
		// This only runs on pages that have dimension data (send_page_view: false was set in config)
		echo "\n\t\tgtag('event', 'page_view', " . wp_json_encode( $params ) . ");";
	}

	/**
	 * Build GA4 event parameters from tracked dimensions.
	 *
	 * Collects all available dimension values and formats them as GA4 event parameters.
	 * Only includes non-empty values. Uses 'wpa_' prefix for custom parameters.
	 *
	 * @since 7.0.4
	 * @return array Associative array of dimension parameters for GA4 event.
	 */
	public function wpa_get_ga4_event_params() {
		$params = array();
		
		// Get configured dimensions to build params based on what's actually enabled
		$current_dimensions = $GLOBALS['WP_ANALYTIFY']->settings->get_option( 'analytiy_custom_dimensions', 'wp-analytify-custom-dimensions' );
		$defined_dimensions = Analytify_Google_Dimensions::get_current_dimensions();
		
		if ( empty( $current_dimensions ) || ! is_array( $current_dimensions ) ) {
			return $params;
		}

		// Iterate through configured dimensions and build params
		foreach ( $current_dimensions as $key => $value ) {
			$type       = isset( $value['type'] ) ? $value['type'] : '';
			$identifier = isset( $defined_dimensions[ $type ]['value'] ) ? $defined_dimensions[ $type ]['value'] : false;

			if ( ! $identifier ) {
				continue;
			}

			$tracking_val = $this->wpa_get_tracking_value( $type );

			// For logged_in, 'false' is a valid value (user not logged in)
			// Include it even if the value is the string 'false'
			if ( 'logged_in' === $type ) {
				// Always include logged_in if it's configured, even if value is 'false'
				$params[ $identifier ] = $tracking_val;
			} elseif ( ! empty( $tracking_val ) ) {
				// For other dimensions, only include if value is not empty
				$params[ $identifier ] = $tracking_val;
			}
		}

		return $params;
	}
}

// Initialize Analytify_Dimensions_Tracking instance.
$wpa_dimensions_tracking = new Analytify_Dimensions_Tracking();
