<?php
// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * A class that contains helper methods for the forms add-on.
 */
class AnalytifyAddonFormHelper {

	/**
	 * Returns the Form Title based on their ID and event category
	 *
	 * @param int    $id             Form ID.
	 * @param string $event_category Event Category.
	 * @return int|string
	 */
	public static function get_form_name( $id, $event_category ) {

		if ( 'analytify_form_wpcf7' === $event_category && is_callable( 'WPCF7_ContactForm::get_instance' ) ) {

			// For CF7.
			$wpcf7 = WPCF7_ContactForm::get_instance( $id );
			if ( $wpcf7 && $wpcf7->title() ) {
				return $wpcf7->title();
			}
		} elseif ( 'analytify_form_wpforms' === $event_category ) {

			// For WPForms.
			if ( get_post( $id ) ) {
				return get_the_title( $id );
			}
		} elseif ( 'analytify_form_ninja_forms' === $event_category && method_exists( 'Ninja_Forms', 'get_setting' ) ) {

			// For Ninja Forms.
			return Ninja_Forms()->form( $id )->get()->get_setting( 'title' );
		} elseif ( 'analytify_form_gf' === $event_category && is_callable( 'GFAPI::get_form' ) ) {

			// For Gravity Forms.
			$gravity_form = GFAPI::get_form( $id );
			if ( $gravity_form && isset( $gravity_form['title'] ) && ! empty( $gravity_form['title'] ) ) {
				return $gravity_form['title'];
			}
		} elseif ( 'analytify_form_formidable' === $event_category && is_callable( 'FrmForm::getName' ) ) {

			// FormFormidable Forms.
			$formidable_name = FrmForm::getName( $id );
			if ( isset( $formidable_name ) && ! empty( $formidable_name ) ) {
				return $formidable_name;
			}
		} elseif ( 'analytify_form_fluentforms' === $event_category && function_exists( 'wpFluent' ) ) {

			// Fluent Forms.
			$fluent_form = wpFluent()->table( 'fluentform_forms' )->find( $id );
			if ( $fluent_form && isset( $fluent_form->title ) && ! empty( $fluent_form->title ) ) {
				return $fluent_form->title;
			}
		} elseif ( 'analytify_form_custom' === $event_category ) {

			// For Custom Forms, the ID is the form name/identifier itself (from name or id attribute).
			// Return it as-is, don't convert to integer.
			return $id;
		}

		// For other form types that don't match above, return ID as integer (fallback).
		return (int) $id;
	}

	/**
	 * All the supported forms, except custom.
	 *
	 * @return array
	 */
	public static function get_supported_forms() {
		return array(
			array(
				'title'          => 'Contact Form 7',
				'option_name'    => 'track_cf7',
				'plugin_file'    => 'contact-form-7/wp-contact-form-7.php',
				'event_category' => 'analytify_form_wpcf7',
			),
			array(
				'title'          => 'WPForms',
				'option_name'    => 'track_wpforms',
				'plugin_file'    => 'wpforms-lite/wpforms.php',
				'event_category' => 'analytify_form_wpforms',
			),
			array(
				'title'          => 'Ninja Forms',
				'option_name'    => 'track_ninjaforms',
				'plugin_file'    => 'ninja-forms/ninja-forms.php',
				'event_category' => 'analytify_form_ninja_forms',
			),
			array(
				'title'          => 'Gravity Forms',
				'option_name'    => 'track_gravityforms',
				'plugin_file'    => 'gravityforms/gravityforms.php',
				'event_category' => 'analytify_form_gf',
			),
			array(
				'title'          => 'Formidable Forms',
				'option_name'    => 'track_formidable',
				'plugin_file'    => 'formidable/formidable.php',
				'event_category' => 'analytify_form_formidable',
			),
			array(
				'title'          => 'Fluent Forms',
				'option_name'    => 'track_fluentforms',
				'plugin_file'    => 'fluentform/fluentform.php',
				'event_category' => 'analytify_form_fluentforms',
			),
		);
	}

	/**
	 * Returns forms that should be displayed in the dashboard.
	 *
	 * @return array
	 */
	public static function get_forms_for_dashboard() {

		// Holds all the forms that have tracking enabled from settings and have active corresponding plugins.
		$analytify_tracking_forms = array();

		// All the supported forms, except custom.
		$registered_forms = self::get_supported_forms();

		foreach ( $registered_forms as $form ) {

			// Compatibility check added for wpforms.
			if ( 'WPForms' === $form['title'] && 'on' === $GLOBALS['WP_ANALYTIFY']->settings->get_option( $form['option_name'], 'wp-analytify-forms' ) && ( is_plugin_active( 'wpforms-lite/wpforms.php' ) || is_plugin_active( 'wpforms/wpforms.php' ) ) ) {
				$analytify_tracking_forms[] = array(
					'title'    => $form['title'],
					'category' => $form['event_category'],
				);
				continue;
			}

			if ( 'on' === $GLOBALS['WP_ANALYTIFY']->settings->get_option( $form['option_name'], 'wp-analytify-forms' ) && is_plugin_active( $form['plugin_file'] ) ) {
				$analytify_tracking_forms[] = array(
					'title'    => $form['title'],
					'category' => $form['event_category'],
				);
			}
		}

		// Check for custom forms.
		if ( 'on' === $GLOBALS['WP_ANALYTIFY']->settings->get_option( 'custom_form', 'wp-analytify-forms' ) ) {
			$analytify_tracking_forms[] = array(
				'title'    => 'Custom Forms',
				'category' => 'analytify_form_custom',
			);
		}

		return $analytify_tracking_forms;
	}
}

// Formate stat rows.
add_filter( 'analytify_forms_row_format', 'analytify_forms_addon_row_formatter', 10, 5 );
/**
 * Formats a single api return to the stats array.
 *
 * @param array   $stats The main stats array.
 * @param string  $_ct   Event Category.
 * @param string  $_lb   Event Label.
 * @param string  $_ac   Event Action.
 * @param integer $count View count.
 *
 * @return array
 */
function analytify_forms_addon_row_formatter( $stats, $_ct, $_lb, $_ac, $count = 0 ) {
	if ( '(not set)' !== $_lb ) {

		// Add ID.
		if ( ! isset( $stats[ $_ct ]['stats'][ $_lb ]['id'] ) && 'analytify_form_custom' !== $_ct ) {
			$stats[ $_ct ]['stats'][ $_lb ]['id'] = $_lb;
		}

		// Add title, if missing.
		if ( ! isset( $stats[ $_ct ]['stats'][ $_lb ]['title'] ) ) {
			$stats[ $_ct ]['stats'][ $_lb ]['title'] = 'analytify_form_custom' === $_ct ? $_lb : AnalytifyAddonFormHelper::get_form_name( $_lb, $_ct );
		}

		// Add default 'impression' and 'submission'.
		if ( ! isset( $stats[ $_ct ]['stats'][ $_lb ]['impression'] ) ) {
			$stats[ $_ct ]['stats'][ $_lb ]['impression'] = 0;
		}
		if ( ! isset( $stats[ $_ct ]['stats'][ $_lb ]['submission'] ) ) {
			$stats[ $_ct ]['stats'][ $_lb ]['submission'] = 0;
		}

		// Add views for submission and impression.
		$stats[ $_ct ]['stats'][ $_lb ][ $_ac ] = isset( $stats[ $_ct ]['stats'][ $_lb ][ $_ac ] ) ? $stats[ $_ct ]['stats'][ $_lb ][ $_ac ] + $count : $count;
	}
	return $stats;
}
