<?php
// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handle Analytify REST end points
 */
class Analytify_Addon_Forms_Rest_API {

	/**
	 * The single instance of the class.
	 *
	 * @var object
	 */
	private static $instance;

	/**
	 * The main Analytify object.
	 *
	 * @var object
	 */
	private $wp_analytify;

	/**
	 * GA version (ga4 or ga3).
	 *
	 * @var string
	 */
	private $ga_mode;

	/**
	 * Selected 'start state'.
	 *
	 * @var string
	 */
	private $start_date;

	/**
	 * Selected 'End state'.
	 *
	 * @var string
	 */
	private $end_date;

	/**
	 * 'info_message' for forms, its returned with response body.
	 *
	 * @var string
	 */
	private $info_message;

	/**
	 * Returns the single instance of the class.
	 *
	 * @return object Class instance
	 */
	public static function get_instance() {
		if ( empty( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Class constructor.
	 */
	private function __construct() {

		// Register API endpoints.
		add_action( 'rest_api_init', array( $this, 'rest_api_init' ) );

	}

	/**
	 * Register end point.
	 */
	public function rest_api_init() {

		$this->wp_analytify = $GLOBALS['WP_ANALYTIFY'];
		$this->ga_mode      = method_exists( 'WPANALYTIFY_Utils', 'get_ga_mode' ) ? WPANALYTIFY_Utils::get_ga_mode() : 'ga3';

		register_rest_route(
			'wp-analytify/v1',
			'/get_forms_report/(?P<request_type>[a-zA-Z0-9-]+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE, // Get Request.
					'callback'            => array( $this, 'handle_request' ),
					'permission_callback' => array( $this, 'permission_check' ),
				),
			)
		);
	}

	private function get_form_title_by_id( $form_id ) {
		// Assuming you have access to the Formidable Forms API or database
		if ( class_exists( 'FrmForm' ) ) {
			$form = FrmForm::getOne( $form_id ); // Fetch form by ID
			if ( $form ) {
				return $form->name; // Return the form title
			}
		}
		return false; // Return false if form not found
	}

	/**
	 * Checks access permission.
	 * Checks if the user is logged-in and checks of the user role has access.
	 *
	 * @return boolean
	 */
	public function permission_check() {
		$is_access_level = $this->wp_analytify->settings->get_option( 'show_analytics_roles_dashboard', 'wp-analytify-dashboard', array( 'administrator' ) );
		return (bool) $this->wp_analytify->pa_check_roles( $is_access_level );
	}

	/**
	 * Handle the request.
	 *
	 * @param WP_REST_Request $request WP Request object.
	 *
	 * @return array|WP_Error
	 */
	public function handle_request( WP_REST_Request $request ) {

		$request_type = $request->get_param( 'request_type' );

		$this->start_date = $request->get_param( 'sd' );
		$this->end_date   = $request->get_param( 'ed' );

		$this->info_message = esc_html__( '\'Impressions\' show how many times a particular form has been viewed.', 'wp-analytify-forms' );

		switch ( $request_type ) {
			case 'main-dashboard':
				$forms = $this->main_dashboard_ga4();
				return array(
					'success' => is_array( $forms ),
					'headers' => array(
						'id' => array(
							'label'    => esc_html__( 'ID', 'wp-analytify-forms' ),
							'th_class' => 'analytify_value_row',
							'td_class' => 'analytify_txt_center',
						),
						'title'      => array(
							'label'    => esc_html__( 'Name', 'wp-analytify-forms' ),
							'th_class' => 'analytify_txt_left',
							'td_class' => '',
						),
						'impression' => array(
							'label'    => esc_html__( 'Impressions', 'wp-analytify-forms' ),
							'th_class' => 'analytify_value_row',
							'td_class' => 'analytify_txt_center',
						),
						'submission' => array(
							'label'    => esc_html__( 'Submissions', 'wp-analytify-forms' ),
							'th_class' => 'analytify_value_row',
							'td_class' => 'analytify_txt_center',
						),
						'conversion_rate' => array(
							'label'    => esc_html__( 'Conversion Rate', 'wp-analytify-forms' ),
							'th_class' => 'analytify_value_row',
							'td_class' => 'analytify_txt_center',
						)
					),
					'stats'   => $forms,
				);
		}
		
		// If no request type match, Return error.
		return new WP_Error( 'analytify_invalid_endpoint', esc_html__( 'Invalid endpoint.', 'wp-analytify' ), array( 'status' => 404 ) );
	}

	/**
	 * Returns formatted stats for dashboard of a GA4 property.
	 *
	 * @return array
	 */
	private function main_dashboard_ga4() {
		// API request limit.
		$api_request_limit = apply_filters('analytify_api_limit_forms_addon', 100, $this->ga_mode);
	
		// Data container.
		$stats = array();
	
		// Build filters for API call and $stats base, based on forms that are available for dashboard.
		$report_filter = array();
		foreach (AnalytifyAddonFormHelper::get_forms_for_dashboard() as $form) {
			// Stats base.
			$stats[$form['category']] = array(
				'stats' => array(),
				'info_message' => $this->info_message,
			);
		}
	
		// Fetch report.
		$raw_report = $this->wp_analytify->get_reports(
			'show-analytify-forms-dashboard',
			array('eventCount'),
			array(
				'start' => $this->start_date,
				'end' => $this->end_date,
			),
			array(
				'customEvent:wpa_category',
				'customEvent:wpa_link_action',
				'customEvent:wpa_link_label',
			),
			array(
				'type' => 'metric',
				'name' => 'eventCount',
				'order' => 'desc',
			),
			array(
				'logic' => 'OR',
				'filters' => $report_filter,
			),
			$api_request_limit,
			true,
		);
	
		// Process the report to calculate impressions, submissions, and conversion rate.
		if (isset($raw_report['rows']) && !empty($raw_report['rows'])) {
			foreach ($raw_report['rows'] as $raw_rows) {
				$category = $raw_rows['customEvent:wpa_category'];
				$action = $raw_rows['customEvent:wpa_link_action'];
				$form_id = $raw_rows['customEvent:wpa_link_label']; // This holds the form ID
				$title = AnalytifyAddonFormHelper::get_form_name( $form_id, $category ); // Fetch form title using helper function

				$count = $raw_rows['eventCount'];
	
				// Initialize if not set
				if (!isset($stats[$category]['stats'][$title])) {
					// Build array in the same order as headers: id, title, impression, submission, conversion_rate
					if ('analytify_form_custom' !== $category) {
						// For non-custom forms, include ID first to match header order
						$stats[$category]['stats'][$title] = array(
							'id' => $form_id,
							'title' => $title,
							'impression' => 0,
							'submission' => 0,
							'conversion_rate' => '0%',
						);
					} else {
						// For custom forms, no ID field
						$stats[$category]['stats'][$title] = array(
							'title' => $title,
							'impression' => 0,
							'submission' => 0,
							'conversion_rate' => '0%',
						);
					}
				}
				
				// Ensure ID is always set for non-custom forms
				if ('analytify_form_custom' !== $category) {
					if (!isset($stats[$category]['stats'][$title]['id'])) {
						$stats[$category]['stats'][$title]['id'] = $form_id;
					}
				}
	
				// Calculate impressions and submissions.
				if ($action === 'impression') {
					$stats[$category]['stats'][$title]['impression'] += $count;
				} elseif ($action === 'submission') {
					$stats[$category]['stats'][$title]['submission'] += $count;
				}
	
				// Calculate conversion rate if there are impressions.
				if ($stats[$category]['stats'][$title]['impression'] > 0) {
					$impressions = $stats[$category]['stats'][$title]['impression'];
					$submissions = $stats[$category]['stats'][$title]['submission'];
					$conversion_rate = ($submissions / $impressions) * 100;
					$stats[$category]['stats'][$title]['conversion_rate'] = round($conversion_rate, 2) . '%';
				}
			}
		}

		// Final step: Ensure ID is first field for non-custom forms to match header order
		foreach ($stats as $category => $category_data) {
			if ('analytify_form_custom' !== $category && isset($category_data['stats']) && is_array($category_data['stats'])) {
				foreach ($category_data['stats'] as $form_key => $form_data) {
					if (isset($form_data['id'])) {
						$id_value = $form_data['id'];
						unset($stats[$category]['stats'][$form_key]['id']);
						$stats[$category]['stats'][$form_key] = array_merge(
							array('id' => $id_value),
							$stats[$category]['stats'][$form_key]
						);
					}
				}
			}
		}

		return $stats;
	}

}

/**
 * Init the instance.
 *
 */
Analytify_Addon_Forms_Rest_API::get_instance();
