<?php
/**
 * Handle Analytify REST end points
 */
class Analytify_Addon_Author_Rest_API {

	/**
	 * The single instance of the class.
	 *
	 * @var object
	 */
	private static $instance;

	/**
	 * The main Analytify object.
	 *
	 * @var object
	 */
	private $wp_analytify;

	/**
	 * GA version (ga4 or ga3).
	 *
	 * @var string
	 */
	private $ga_mode;

	/**
	 * Selected 'start state'.
	 *
	 * @var string
	 */
	private $start_date;

	/**
	 * Selected 'End state'.
	 *
	 * @var string
	 */
	private $end_date;

	/**
	 * Selected 'author id'.
	 *
	 * @var null|int
	 */
	private $author_id = null;

	/**
	 * Returns the single instance of the class.
	 *
	 * @return object Class instance
	 */
	public static function get_instance() {
		if ( empty( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Class constructor.
	 *
	 * @return void
	 */
	private function __construct() {

		// Register API endpoints.
		add_action( 'rest_api_init', array( $this, 'rest_api_init' ) );
	}

	/**
	 * Register end point.
	 *
	 * @return void
	 */
	public function rest_api_init() {

		$this->wp_analytify = $GLOBALS['WP_ANALYTIFY'];
		$this->ga_mode      = method_exists( 'WPANALYTIFY_Utils', 'get_ga_mode' ) ? WPANALYTIFY_Utils::get_ga_mode() : 'ga3';

		register_rest_route(
			'wp-analytify/v1',
			'/get_author_report/(?P<request_type>[a-zA-Z0-9-]+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE, // Get Request.
					'callback'            => array( $this, 'handle_request' ),
					'permission_callback' => array( $this, 'permission_check' ),
				),
			)
		);
	}

	/**
	 * Checks access permission.
	 * Checks if the user is logged-in and checks of the user role has access.
	 *
	 * @return boolean
	 */
	public function permission_check() {
		$is_access_level = $this->wp_analytify->settings->get_option( 'show_analytics_roles_dashboard', 'wp-analytify-dashboard', array( 'administrator' ) );
		return (bool) $this->wp_analytify->pa_check_roles( $is_access_level );
	}

	/**
	 * Handles the request.
	 *
	 * @param WP_REST_Request $request WP REST request object.
	 *
	 * @return array|WP_Error
	 */
	public function handle_request( WP_REST_Request $request ) {

		$request_type = $request->get_param( 'request_type' );

		$this->start_date = $request->get_param( 'sd' );
		$this->end_date   = $request->get_param( 'ed' );
		$this->author_id  = $request->get_param( 'author_id' );

		switch ( $request_type ) {
			case 'main-dashboard':
				return $this->main_dashboard();
		}

		// If no request type match, Return error.
		return new WP_Error( 'analytify_invalid_endpoint', __( 'Invalid endpoint.', 'wp-analytify-authors' ), array( 'status' => 404 ) );
	}


	/**
	 * Endpoint for 'main dashboard'.
	 *
	 * @return array
	 */
	public function main_dashboard() {

		if ( is_null( $this->author_id ) ) {
			return new WP_Error( 'analytify_invalid_author_id', __( 'Invalid author id.', 'wp-analytify-authors' ), array( 'status' => 404 ) );
		}

		if ( '0' != $this->author_id ) {
			$user = get_user_by( 'id', $this->author_id );
			if ( ! $user ) {
				return new WP_Error( 'analytify_invalid_author_id', __( 'Invalid author id.', 'wp-analytify-authors' ), array( 'status' => 404 ) );
			}
			$author_display_name = $user->display_name;
		} else {
			$author_display_name = esc_html__( 'all authors', 'wp-analytify-authors' );
		}

		$success   = true;
		$error_box = false;
		$stats     = array();
		$headers   = array(
			'no'          => array(
				'label'    => esc_html__( '#', 'wp-analytify-authors' ),
				'type'     => 'counter',
				'th_class' => 'analytify_num_row',
				'td_class' => 'analytify_txt_center',
			),
			'post'       => array(
				'label'    => esc_html__( 'Post', 'wp-analytify-authors' ),
				'th_class' => 'analytify_txt_left',
				'td_class' => '',
			),
			'author' => array(
				'label'    => esc_html__( 'Author', 'wp-analytify-authors' ),
				'th_class' => 'analytify_value_row',
				'td_class' => 'analytify_txt_center',
			),
			'sessions'    => array(
				'label'    => esc_html__( 'Sessions', 'wp-analytify-authors' ),
				'th_class' => 'analytify_value_row',
				'td_class' => 'analytify_txt_center',
			),
			'page_views'  => array(
				'label'    => esc_html__( 'Page Views', 'wp-analytify-authors' ),
				'th_class' => 'analytify_value_row',
				'td_class' => 'analytify_txt_center',
			),
			'visitors'    => array(
				'label'    => esc_html__( 'Visitors', 'wp-analytify-authors' ),
				'th_class' => 'analytify_value_row',
				'td_class' => 'analytify_txt_center',
			),
			'time'    => array(
				'label'    => esc_html__( 'Avg. Time', 'wp-analytify-authors' ),
				'th_class' => 'analytify_value_row',
				'td_class' => 'analytify_txt_center',
			),
			'bounce_rate' => array(
				'label'    => esc_html__( 'Bounce Rate', 'wp-analytify-authors' ),
				'th_class' => 'analytify_value_row',
				'td_class' => 'analytify_txt_center',
			),
		);

		$check_for_author_dimension = $this->check_for_author_dimension();

		$api_limit = apply_filters( 'analytify_api_limit_authors_main_dashboard', 50, 'dashboard', $this->author_id );

		if ( is_array( $check_for_author_dimension ) ) {
			$success   = false;
			$error_box = $check_for_author_dimension;
		} else {
			if ( 'ga4' === $this->ga_mode ) {

				if ( '0' != $this->author_id ) {
					$dimension_filers[] = array(
						'type'       => 'dimension',
						'name'       => 'customEvent:wpa_author',
						'match_type' => '4',
						'value'      => $user->user_login,
					);
					unset( $headers['author'] );
				} else {
					$dimension_filers[] = array(
						'type'           => 'dimension',
						'name'           => 'customEvent:wpa_author',
						'match_type'     => 4,
						'value'          => '(not set)',
						'not_expression' => true,
					);
				}

				$stats_raw = $this->wp_analytify->get_reports(
					'show-authors-stats-' . $this->author_id,
					array(
						'sessions',
						'screenPageViews',
						'totalUsers',
						'userEngagementDuration',
						'bounceRate',
					),
					$this->get_dates(),
					array(
						'customEvent:wpa_author',
						'PageTitle',
						'pagePath',
					),
					array(
						'type'  => 'metric',
						'name'  => 'screenPageViews',
						'order' => 'desc',
					),
					array(
						'logic'   => 'AND',
						'filters' => $dimension_filers,
					),
					$api_limit,
					true
				);

				if ( isset( $stats_raw['rows'] ) && $stats_raw['rows'] ) {
					foreach ( $stats_raw['rows'] as $row ) {
						$single_row['no']   = null;
						$single_row['post'] = '<a href="' . esc_url_raw( $row['pagePath'] ) . '" target="_blank">' . esc_html( $row['PageTitle'] ) . '</a>';
						if ( '0' == $this->author_id ) {
							$single_row['author'] = $row['customEvent:wpa_author'];
						}
						$single_row['sessions']    = $row['sessions'];
						$single_row['page_views']  = $row['screenPageViews'];
						$single_row['visitors']    = $row['totalUsers'];
						$single_row['time']        = $row['userEngagementDuration'] ? WPANALYTIFY_Utils::pretty_time( $row['userEngagementDuration'] ) : 0;
						$single_row['bounce_rate'] = $row['bounceRate'] ? WPANALYTIFY_Utils::pretty_numbers( $row['bounceRate'] ) . '%' : 0 . '%';

						$stats[] = $single_row;
					}
				}
			}
		}

		// Translators: %s is the error message.
		$default_footer_message = sprintf( 'Top posts by %s.' , $author_display_name );
				
		return array(
			'success'    => $success,
			'headers'    => $success ? $headers : null,
			'stats'      => $stats,
			'error_box'  => $error_box,
			'footer'     => $success ? apply_filters( 'analytify_author_dashboard_footer', $default_footer_message, array( $this->start_date, $this->end_date ), $this->author_id ) : null,
			'pagination' => true,
		);
	}

	/**
	 * Checks that the dimension add-on is active and the author dimension is set.
	 *
	 * @return array|true
	 */
	private function check_for_author_dimension() {

		if ( ! class_exists( 'Analytify_Google_Dimensions' ) ) {
			return $this->error_box_return( 'dimensions-addon' );
		}

		if ( 'ga4' === $this->ga_mode ) {
			$set_dimensions = $this->wp_analytify->settings->get_option( 'analytiy_custom_dimensions', 'wp-analytify-custom-dimensions' );
			if ( is_array( $set_dimensions ) ) {
				foreach ($set_dimensions as $key => $dimension) {
					if ('author' === $dimension['type']) {
						return true;
					}
				}
			}
			return $this->error_box_return( 'dimensions-author' );
		}
	}

	/**
	 * Returns error boxes messages.
	 *
	 * @param string $type Error type.
	 * @return array
	 */
	private function error_box_return( $type ) {
		$error_markup = '';

		if ( 'dimensions-addon' === $type ) {
			$error_markup = '<p class="analytify-promo-popup-paragraph analytify-error-popup-paragraph">' . esc_html__( 'Authors Dashboard can\'t be loaded until you enable the Custom Dimensions addon and setup the Author Custom Dimension.', 'wp-analytify-authors' ) . '</p>';
			$error_markup .= '<p class="analytify-promo-popup-paragraph analytify-error-popup-paragraph"><a href="https://analytify.io/doc/get-started-with-custom-dimensions-addon/" target="_blank">' . esc_html__( 'Click here to read our guide on Custom Dimensions', 'wp-analytify-authors' ) . '</a></p>';
		}

		if ( 'dimensions-author' === $type ) {
			$error_markup = '<p class="analytify-promo-popup-paragraph analytify-error-popup-paragraph">' . esc_html__( 'Authors Dashboard can\'t be loaded until you setup the Author Custom Dimension.', 'wp-analytify-authors' ) . '</p>';
			$error_markup .= '<p class="analytify-promo-popup-paragraph analytify-error-popup-paragraph"><a href="https://analytify.io/doc/get-started-with-custom-dimensions-addon/" target="_blank">' . esc_html__( 'Click here to read our guide on Custom Dimensions', 'wp-analytify-authors' ) . '</a></p>';
		}

		return array(
			'title'   => esc_html__( 'Unable To Fetch Reports', 'wp-analytify-authors' ),
			'content' => $error_markup,
		);
	}

	/**
	 * Returns start and end date as an array to be used for GA4's get_reports()
	 *
	 * @return array
	 */
	private function get_dates() {
		return array(
			'start' => $this->start_date,
			'end'   => $this->end_date,
		);
	}

}

/**
 * Init the instance.
 *
 */
Analytify_Addon_Author_Rest_API::get_instance();
