<?php
ob_start();

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Analytify_Campaigns_Rest_API' ) ) {

	class Analytify_Campaigns_Rest_API {

		/**
		 * The single instance of the class.
		 *
		 * @var object
		 */
		private static $instance;

		/**
		 * The main Analytify object.
		 *
		 * @var object
		 */
		private $wp_analytify;

		/**
		 * GA version (ga4 or ga3).
		 *
		 * @var string
		 */
		private $ga_mode;

		/**
		 * Selected 'start state'.
		 *
		 * @var string
		 */
		private $start_date;

		/**
		 * Selected 'End state'.
		 *
		 * @var string
		 */
		private $end_date;

		/**
		 * Name of the Campaign.
		 *
		 * @var string
		 */
		private $campaign_name;

		/**
		 * Tyoe of the Request.
		 *
		 * @var string
		 */
		private $request_type;

		/**
		 * Returns the single instance of the class.
		 *
		 * @return object Class instance
		 */
		public static function get_instance() {
			if ( empty( self::$instance ) ) {
				self::$instance = new self();
			}

			return self::$instance;
		}

		/**
		 * Class constructor.
		 */
		private function __construct() {
			add_action( 'rest_api_init', array( $this, 'rest_api_init' ) );
		}

		/**
		 * Register end point.
		 */
		public function rest_api_init() {

			$this->wp_analytify = $GLOBALS['WP_ANALYTIFY'];
			$this->ga_mode      = method_exists( 'WPANALYTIFY_Utils', 'get_ga_mode' ) ? WPANALYTIFY_Utils::get_ga_mode() : 'ga3';

			register_rest_route(
				'wp-analytify/v1',
				'/get_pro_campaigns_report/(?P<request_type>[a-zA-Z0-9-]+)',
				array(
					array(
						'methods'             => WP_REST_Server::READABLE,
						'callback'            => array( $this, 'handle_request' ),
						'permission_callback' => array( $this, 'permission_check' ),
					),
				)
			);
		}

		/**
		 * Checks access permission.
		 * Checks if the user is logged-in and checks of the user role has access.
		 *
		 * @return boolean
		 */
		public function permission_check() {
			$is_access_level = $this->wp_analytify->settings->get_option( 'show_analytics_roles_dashboard', 'wp-analytify-dashboard', array( 'administrator' ) );
			return (bool) $this->wp_analytify->pa_check_roles( $is_access_level );
		}

		/**
		 * Handle the request.
		 *
		 * @param WP_REST_Request $request WP Request object,
		 *
		 * @return array|WP_Error
		 */
		public function handle_request( WP_REST_Request $request ) {

			$this->request_type = $request->get_param( 'request_type' );

			$this->start_date    = $request->get_param( 'sd' );
			$this->end_date      = $request->get_param( 'ed' );
			$this->campaign_name = $request->get_param( 'cn' );

			switch ( $this->request_type ) {
				case 'campaigns':
					return $this->campaigns();
				case 'campaigns-detail':
					return $this->campaigns_detail();
			}

			// If no request type match, Return error.
			return new WP_Error( 'analytify_invalid_endpoint', __( 'Invalid endpoint.', 'wp-analytify' ), array( 'status' => 404 ) );
		}

		/**
		 * Returns all the active campaigns.
		 *
		 * @return string
		 */
		private function campaigns() {

			$api_limit = apply_filters( 'analytify_api_limit_campaigns_stats', 50, 'dashboard' );

			if ( 'ga4' === $this->ga_mode ) {
				$stats = $this->wp_analytify->get_reports(
					'show-campaigns-stats',
					array(
						'sessions',
						'newUsers',
						'bounceRate',
						'averageSessionDuration',
						'screenPageViews',
					),
					array(
						'start' => $this->start_date,
						'end'   => $this->end_date,
					),
					array(
						'sessionCampaignName',
					),
					array(
						'type'  => 'metric',
						'name'  => 'sessions',
						'order' => 'desc',
					),
					array(),
					$api_limit
				);

				include ANALYTIFY_PRO_CAMPAIGNS_ADDON_ROOT_PATH . '/views/campaign-stats.php';
			}
		}

		/**
		 * Returns the details for a single campaign.
		 *
		 * @return string
		 */
		private function campaigns_detail() {

			$api_limit = apply_filters( 'analytify_api_limit_campaigns_details_stats', 50, 'dashboard', $this->campaign_name );

			if ( 'ga4' === $this->ga_mode ) {
				$stats = $this->wp_analytify->get_reports(
					'show-campaign-detail-' . $this->campaign_name,
					array(
						'totalUsers',
						'newUsers',
						'sessions',
						'bounceRate',
						'averageSessionDuration',
						'transactions',
						'purchaseRevenue',
						'engagedSessions'
					),
					array(
						'start' => $this->start_date,
						'end'   => $this->end_date,
					),
					array(
						'sessionCampaignName',
						'sessionSourceMedium',
						'sessionManualTerm',  // Add for utm_term
            			'sessionManualAdContent',
					),
					array(
						'type'  => 'metric',
						'name'  => 'transactions',
						'order' => 'desc',
					),
					array(
						'logic'   => 'AND',
						'filters' => array(
							array(
								'type'       => 'dimension',
								'name'       => 'sessionCampaignName',
								'match_type' => 1,
								'value'      => $this->campaign_name,
							),
						),
					),
					$api_limit
				);

				// Check if there are any rows
				$has_rows = ! empty( $stats['rows'] ) && is_array( $stats['rows'] ) && count( $stats['rows'] ) > 0;
				
				if ( ! $has_rows ) {
					// Show "No activity during this period" message
					$html = '<div class="analytify-stats-error-msg">
						<div class="wpb-error-box">
							<span class="blk"><span class="line"></span><span class="dot"></span></span>
							<span class="information-txt">' . esc_html__( 'No activity during this period.', 'wp-analytify-pro' ) . '</span>
						</div>
					</div>';
				} else {
					$html = '<table class="wp_analytify_campaign_paginated" data-product-per-page="10">
					<thead>
						<tr>
							<th>Source/Medium</th>
							<th>Term</th> 
							<th>Content</th> 
							<th>Users</th>
							<th>New Users</th>
							<th>Sessions</th>
							<th>Engaged Sessions</th>
							<th>Bounce Rate</th>
							<th>Avg. Session Duration</th>
							<th>E-commerce Conversion Rate</th>
							<th>Transactions</th>
							<th>Revenue</th>
						</tr></thead><tbody>';
					foreach ( $stats['rows'] as $key => $data ) {
						$html .= '<tr>
								<td>' . WPANALYTIFY_Utils::pretty_numbers( $data['sessionSourceMedium'] ) . '</td>
								<td>' . ( !empty($data['sessionManualTerm']) ? WPANALYTIFY_Utils::pretty_numbers( $data['sessionManualTerm'] ) : 'N/A' ) . '</td> 
								<td>' . ( !empty($data['sessionManualAdContent']) ? WPANALYTIFY_Utils::pretty_numbers( $data['sessionManualAdContent'] ) : 'N/A' ) . '</td> 
								<td>' . WPANALYTIFY_Utils::pretty_numbers( $data['totalUsers'] ) . '</td>
								<td>' . WPANALYTIFY_Utils::pretty_numbers( $data['newUsers'] ) . '</td>
								<td>' . WPANALYTIFY_Utils::pretty_numbers( $data['sessions'] ) . '</td>
								<td>' . WPANALYTIFY_Utils::pretty_numbers( $data['engagedSessions'] ) . '</td>
								<td>' . number_format( $data['bounceRate'], 2 ) . '%</td>
								<td>' . WPANALYTIFY_Utils::pretty_time( $data['averageSessionDuration'] ) . '</td>
								<td>' . WPANALYTIFY_Utils::pretty_numbers( round($data['transactions']/$data['sessions']*100, 2) ) . '%</td>
								<td>' . WPANALYTIFY_Utils::pretty_numbers( $data['transactions'] ) . '%</td>
								<td>' . WPANALYTIFY_Utils::pretty_numbers( $data['purchaseRevenue'] ) . '</td>
							</tr>';
					}
					$html .= '</tbody></table>';
				}

				// Only add pagination wrapper if there are rows
				if ( $has_rows ) {
					$html .= '<div class="wp_analytify_campaign_pagination_wraper"></div>';
				}
			}

			wp_send_json(
				array(
					'body' => $html,
				)
			);
		}
	}

	Analytify_Campaigns_Rest_API::get_instance();
}

ob_end_flush();