<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Track WooCommerce Subscriptions, only for the subscriptions plugin built by WooThemes.
 * https://woocommerce.com/products/woocommerce-subscriptions/
 *
 * @since 3.0
 */
class Analytify_Woo_Subscription {

	/**
	 * Constructor method.
	 */
	function __construct() {

		add_filter( 'analytify_do_not_track_completed_purchase', array( $this, 'set_subscription_order_identity' ), 10, 2 );
		add_action( 'subscriptions_activated_for_order', array( $this, 'track_subscription_activate' ), 10, 1 );
		add_action( 'woocommerce_subscription_status_on-hold_to_active', array( $this, 'track_reactivated_subscription' ) );
		add_action(	'woocommerce_subscription_status_on-hold', array( $this, 'track_suspend_subscription' ) );
		add_action( 'woocommerce_subscription_status_cancelled', array( $this, 'track_cancelled_subscription' ) );
		add_action( 'woocommerce_scheduled_subscription_trial_end', array( $this , 'track_trail_ended' ) );
		add_action( 'woocommerce_scheduled_subscription_end_of_prepaid_term', array( $this, 'track_end_of_prepaid'	) );
		add_action( 'woocommerce_scheduled_subscription_expiration', array( $this, 'track_expire_subscription' ) );
	}

	/**
	 * Get subscription identity.
	 *
	 * @param object $track_completed_purchase
	 * @param int $order_id
	 * @return object
	 */
	public function set_subscription_order_identity( $track_completed_purchase, $order_id ) {

		$subscriptions = wcs_get_subscriptions_for_renewal_order( $order_id );

		if ( ! empty( $subscriptions ) ) {
			foreach ( $subscriptions as $subscription ) {
				$subscription_id  = $subscription->get_id(); // order id
				$subscription_cid = get_post_meta( $order_id, 'analytify_woo_cid', true );

				if ( empty( $subscription_cid ) ) {
					$subscription_cid = $GLOBALS['WPAnalytifyProWooCommerce']->get_cid( $order_id );
					$GLOBALS['WPAnalytifyProWooCommerce']->store_ga_identity( $order_id, $subscription_cid );
				}

				$related_orders = $subscription->get_related_orders( 'ids' );

				if ( $subscription_cid && ! empty( $related_orders ) ) {
					foreach ( $related_orders as $related_order_id ) {
						$related_order_cid = get_post_meta( $related_order_id, 'analytify_woo_cid', true );

						if ( empty( $related_order_cid ) ) {
						  $subscription_cid = $GLOBALS['WPAnalytifyProWooCommerce']->get_cid( $related_order_cid );
						  $GLOBALS['WPAnalytifyProWooCommerce']->store_ga_identity( $related_order_id, $subscription_cid );
						}
					}
				}
			}
		}

		return $track_completed_purchase;
	}

	/**
	* Track when subscription complete.
	*
	* @param int $order_id
	* @since 3.0
	*/
	public function track_subscription_activate( $order ) {

		if ( ! $order instanceof WC_Order ) {
			$order = wc_get_order( $order );
		}

		$subscriptions = wcs_get_subscriptions_for_order( $order );

		if ( empty( $subscriptions ) ) {
			return;
		}

		foreach ( $subscriptions as $subscription ) {

			$cid = $GLOBALS['WPAnalytifyProWooCommerce']->get_cid( $subscription->get_id() );

			if ( method_exists( 'WPANALYTIFY_Utils', 'get_ga_mode' ) && 'ga4' === WPANALYTIFY_Utils::get_ga_mode() ) {

				$event[] = array(
					'name'   => 'subscription_activated',
					'params' => array(
						'wpa_category' => 'subscription_id',
						'wpa_label'    => $subscription->get_id(),
					),
				);

				analytify_mp_ga4( $event );

			} else {

				$attr = array(
					't'   => 'event',
					'ec'  => 'Tracking by Analytify Woo Subscription', // category
					'ea'  => 'subscription activated',
					'el'  => $subscription->get_id(),
					'cid' => $cid,
				);

				$GLOBALS['WPAnalytifyProWooCommerce']->hit( $attr );

			}

		}
	}

	/**
	 * Track when user resubscribe.
	 *
	 * @since 3.0
	 */
	public function track_reactivated_subscription( $subscription ) {

		$cid = $GLOBALS['WPAnalytifyProWooCommerce']->get_cid( $subscription->get_id() );

		if ( method_exists( 'WPANALYTIFY_Utils', 'get_ga_mode' ) && 'ga4' === WPANALYTIFY_Utils::get_ga_mode() ) {

			$event[] = array(
				'name'   => 'subscription_reactivated',
				'params' => array(
					'wpa_category' => 'subscription_id',
					'wpa_label'    => $subscription->get_id(),
					'wpa_user_id'  => $subscription->get_user_id(),
				),
			);

			analytify_mp_ga4( $event );

		} else {

			$attr = array(
				't'   => 'event',
				'ec'  => 'Tracking by Analytify Woo Subscription', // category
				'ea'  => 'subscription reactivated',
				'el'  => $subscription->get_id(),
				'uid' => $subscription->get_user_id(),
				'ni'  => 1,
				'cid' => $cid,

			);

			$GLOBALS['WPAnalytifyProWooCommerce']->hit( $attr );
		}
	}

	/**
	 * Track when subscription suspend.
	 *
	 * @since 3.0
	 */
	public function track_suspend_subscription( $subscription ) {

		$cid = $GLOBALS['WPAnalytifyProWooCommerce']->get_cid( $subscription->get_id() );

		if ( method_exists( 'WPANALYTIFY_Utils', 'get_ga_mode' ) && 'ga4' === WPANALYTIFY_Utils::get_ga_mode() ) {

			$event[] = array(
				'name'   => 'subscription_suspended',
				'params' => array(
					'wpa_category' => 'subscription_id',
					'wpa_label'    => $subscription->get_id(),
					'wpa_user_id'  => $subscription->get_user_id(),
				),
			);

			analytify_mp_ga4( $event );

		} else {

			$attr = array(
				't'   => 'event',
				'ec'  => 'Tracking by Analytify Woo Subscription', // category
				'ea'  => 'subscription suspended',
				'el'  => $subscription->get_id(),
				'uid' => $subscription->get_user_id(),
				'ni'  => 1,
				'cid' => $cid

			);

			$GLOBALS['WPAnalytifyProWooCommerce']->hit( $attr );
		}
	}

	/**
	 * Track when subscription cancelled.
	 *
	 * @since 3.0
	 */
	public function track_cancelled_subscription( $subscription ) {

		$cid = $GLOBALS['WPAnalytifyProWooCommerce']->get_cid( $subscription->get_id() );

		if ( method_exists( 'WPANALYTIFY_Utils', 'get_ga_mode' ) && 'ga4' === WPANALYTIFY_Utils::get_ga_mode() ) {

			$event[] = array(
				'name'   => 'subscription_cancelled',
				'params' => array(
					'wpa_category' => 'subscription_id',
					'wpa_label'    => $subscription->get_id(),
					'wpa_user_id'  => $subscription->get_user_id(),
				),
			);

			analytify_mp_ga4( $event );

		} else {

			$attr = array(
				't'   => 'event',
				'ec'  => 'Tracking by Analytify Woo Subscription', // category
				'ea'  => 'subscription cancelled',
				'el'  => $subscription->get_id(),
				'uid' => $subscription->get_user_id(),
				'ni'  => 1,
				'cid' => $cid

			);

			$GLOBALS['WPAnalytifyProWooCommerce']->hit( $attr );
		}
	}

	/**
	 * Track trail conversion.
	 *
	 * @since 3.0
	 */
	public function track_trail_ended( $subscription_id ) {

		$subscription = wcs_get_subscription( $subscription_id );
		$cid = $GLOBALS['WPAnalytifyProWooCommerce']->get_cid( $subscription_id );

		if ( method_exists( 'WPANALYTIFY_Utils', 'get_ga_mode' ) && 'ga4' === WPANALYTIFY_Utils::get_ga_mode() ) {

			$event[] = array(
				'name'   => 'subscription_trial_ended',
				'params' => array(
					'subscription_id' => $subscription->get_id(),
					'user_id'         => $subscription->get_user_id(),
					'cid'             => $cid,
				),
			);

			$trial_conversion_event_name = ( $subscription->get_completed_payment_count() > 1 ) ? 'subscription_trial_converted' : 'subscription_trial_cancelled';

			$event[] = array(
				'name'   => $trial_conversion_event_name,
				'params' => array(
					'wpa_category' => 'subscription_id',
					'wpa_label'    => $subscription->get_id(),
					'wpa_user_id'  => $subscription->get_user_id(),
				),
			);

			analytify_mp_ga4( $event );

		} else {
			$attr = array(
				't'   => 'event',
				'ec'  => 'Tracking by Analytify Woo Subscription', // category
				'ea'  => 'subscription trial ended',
				'el'  => $subscription->get_id(),
				'uid' => $subscription->get_user_id(),
				'ni'  => 1,
				'cid' => $cid
			);

			$GLOBALS['WPAnalytifyProWooCommerce']->hit( $attr );

			// Handle trial conversions - if a subscription has more than a single completed payment, assume it converted
			if ( $subscription->get_completed_payment_count() > 1 ) {
				$attr = array(
					't'   => 'event',
					'ec'  => 'Tracking by Analytify Woo Subscription', // category
					'ea'  => 'subscription trial converted',
					'el'  => $subscription->get_id(),
					'uid' => $subscription->get_user_id(),
					'ni'  => 1,
					'cid' => $cid
	
				);
				$GLOBALS['WPAnalytifyProWooCommerce']->hit( $attr );
			} else {
				$attr = array(
					't'   => 'event',
					'ec'  => 'Tracking by Analytify Woo Subscription', // category
					'ea'  => 'subscription trial cancelled',
					'el'  => $subscription->get_id(),
					'uid' => $subscription->get_user_id(),
					'ni'  => 1,
					'cid' => $cid
	
				);
				$GLOBALS['WPAnalytifyProWooCommerce']->hit( $attr );
			}
		}

	}

	/**
	 * Track when subscription of prepaid term.
	 *
	 * @since 3.0
	 */
	public function track_end_of_prepaid( $subscription_id ) {

		$subscription = wcs_get_subscription( $subscription_id );
		$cid = $GLOBALS['WPAnalytifyProWooCommerce']->get_cid( $subscription_id );

		if ( method_exists( 'WPANALYTIFY_Utils', 'get_ga_mode' ) && 'ga4' === WPANALYTIFY_Utils::get_ga_mode() ) {

			$event[] = array(
				'name'   => 'subscription_end_of_prepaid_term',
				'params' => array(
					'wpa_category' => 'subscription_id',
					'wpa_label'    => $subscription->get_id(),
					'wpa_user_id'  => $subscription->get_user_id(),
				),
			);

			analytify_mp_ga4( $event );

		} else {

			$attr = array(
				't'   => 'event',
				'ec'  => 'Tracking by Analytify Woo Subscription', // category
				'ea'  => 'subscription end of prepaid term',
				'el'  => $subscription->get_id(),
				'uid' => $subscription->get_user_id(),
				'ni'  => 1,
				'cid' => $cid
	
			);
	
			$GLOBALS['WPAnalytifyProWooCommerce']->hit( $attr );
			
		}
	}

	/**
	 * Track when subscription expired.
	 *
	 * @since 3.0
	 */
	public function track_expire_subscription(	$subscription_id ) {

		$subscription = wcs_get_subscription( $subscription_id );
		$cid = $GLOBALS['WPAnalytifyProWooCommerce']->get_cid( $subscription_id );

		if ( method_exists( 'WPANALYTIFY_Utils', 'get_ga_mode' ) && 'ga4' === WPANALYTIFY_Utils::get_ga_mode() ) {

			$event[] = array(
				'name'   => 'subscription_expired',
				'params' => array(
					'wpa_category' => 'subscription_id',
					'wpa_label'    => $subscription->get_id(),
					'wpa_user_id'  => $subscription->get_user_id(),
				),
			);

			analytify_mp_ga4( $event );

		} else {

			$attr = array(
				't'   => 'event',
				'ec'  => 'Tracking by Analytify Woo Subscription', // category
				'ea'  => 'subscription expired',
				'el'  => $subscription->get_id(),
				'uid' => $subscription->get_user_id(),
				'ni'  => 1,
				'cid' => $cid

			);
			$GLOBALS['WPAnalytifyProWooCommerce']->hit( $attr );
		}
	}

}

new Analytify_Woo_Subscription();
