<?php
/**
 * Plugin Name: SityCloud AI Assistant
 * Description: وكيل ذكاء اصطناعي مدمج لمنصة Sity Cloud (بحث – Debug – حماية – تقارير – دعم عملاء "أمنية").
 * Version: 1.1.0
 * Author: Sity Cloud
 */

if (!defined('ABSPATH')) {
    exit;
}

// ======================================================
//  CONFIG: Gemini API KEY (غيّر القيمة بس)
// ======================================================
if (!defined('SITYCLOUD_GEMINI_API_KEY')) {
    define('SITYCLOUD_GEMINI_API_KEY', 'AIzaSyDlk1zhrrBALNAHTWc9S_HckIXRHDWZoj8');
}

// ======================================================
//  MAIN CLASS
// ======================================================
class SityCloud_AI_Assistant {

    public function __construct() {
        add_action('admin_menu', array($this, 'register_admin_menu'));
        add_action('admin_init', array($this, 'maybe_multisite_notice'));

        // Shortcodes (Frontend)
        add_action('init', array($this, 'register_shortcodes'));

        // AJAX actions (Admin)
        add_action('wp_ajax_sitycloud_ai_search_network', array($this, 'ajax_search_network'));
        add_action('wp_ajax_sitycloud_ai_debugger', array($this, 'ajax_ai_debugger'));
        add_action('wp_ajax_sitycloud_ai_support', array($this, 'ajax_ai_support'));
        add_action('wp_ajax_sitycloud_ai_auto_fix', array($this, 'ajax_ai_auto_fix'));
        add_action('wp_ajax_sitycloud_ai_report', array($this, 'ajax_ai_report'));
        add_action('wp_ajax_sitycloud_ai_security_agent', array($this, 'ajax_ai_security_agent'));

        // AJAX actions (Frontend Omnia chat)
        add_action('wp_ajax_sitycloud_ai_omnia_chat', array($this, 'ajax_ai_omnia_chat'));
        add_action('wp_ajax_nopriv_sitycloud_ai_omnia_chat', array($this, 'ajax_ai_omnia_chat'));
    }

    // ==================================================
    //  Admin Notice if not multisite
    // ==================================================
    public function maybe_multisite_notice() {
        if (!is_multisite()) {
            add_action('admin_notices', function () {
                ?>
                <div class="notice notice-info">
                    <p><strong>SityCloud AI Assistant:</strong> يمكن استخدام البلجن على موقع واحد، لكن قوته الكاملة تظهر مع شبكة Sity Cloud (Multisite).</p>
                </div>
                <?php
            });
        }
    }

    // ==================================================
    //  Register Admin Menu
    // ==================================================
    public function register_admin_menu() {
        add_menu_page(
            'SityCloud AI',
            'SityCloud AI',
            'manage_options',
            'sitycloud-ai-assistant',
            array($this, 'render_admin_page'),
            'dashicons-art',
            3
        );
    }

    // ==================================================
    //  Shortcodes (Frontend)
// ==================================================
    public function register_shortcodes() {
        add_shortcode('sitycloud_omnia_chat', array($this, 'render_omnia_chat_shortcode'));
    }

    // ==================================================
    //  Frontend Chatbot "Omnia" Shortcode
    // ==================================================
    public function render_omnia_chat_shortcode($atts = array(), $content = '') {
        $nonce = wp_create_nonce('sitycloud_ai_omnia');
        ob_start();
        ?>
        <div class="sitycloud-omnia-chat-widget">
            <div class="sc-omnia-header">
                <div class="sc-omnia-avatar">أ</div>
                <div class="sc-omnia-meta">
                    <div class="sc-omnia-name">أمنية – دعم عملاء Sity Cloud</div>
                    <div class="sc-omnia-status">متصلة لمساعدتك الآن</div>
                </div>
            </div>
            <div class="sc-omnia-messages" id="sc-omnia-messages">
                <div class="sc-omnia-msg sc-omnia-bot">
                    <div class="sc-omnia-bubble">
                        أهلاً بيك 👋 أنا <strong>أمنية</strong> من فريق Sity Cloud.<br>
                        اسألني عن الباقات، إنشاء موقع، أو أي سؤال يخص حسابك وهساعدك بأبسط شكل.
                    </div>
                </div>
            </div>
            <form class="sc-omnia-input-row" id="sc-omnia-form">
                <input type="text" id="sc-omnia-input" placeholder="اكتب سؤالك هنا..." autocomplete="off">
                <button type="submit">إرسال</button>
            </form>
        </div>

        <style>
            .sitycloud-omnia-chat-widget {
                direction: rtl;
                text-align: right;
                max-width: 420px;
                margin: 20px auto;
                border-radius: 24px;
                background: radial-gradient(circle at top right, #f5f0ff 0, #ffffff 40%, #f7f7ff 100%);
                box-shadow: 0 20px 45px rgba(110, 69, 255, 0.18);
                border: 1px solid rgba(190, 170, 255, 0.6);
                overflow: hidden;
                font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
            }
            .sc-omnia-header {
                display: flex;
                align-items: center;
                gap: 12px;
                padding: 14px 16px;
                background: linear-gradient(135deg, #6b46ff, #c77dff);
                color: #fff;
            }
            .sc-omnia-avatar {
                width: 34px;
                height: 34px;
                border-radius: 50%;
                background: rgba(255, 255, 255, 0.2);
                display: flex;
                align-items: center;
                justify-content: center;
                font-weight: 700;
                font-size: 16px;
            }
            .sc-omnia-meta {
                display: flex;
                flex-direction: column;
                gap: 2px;
            }
            .sc-omnia-name {
                font-size: 14px;
                font-weight: 600;
            }
            .sc-omnia-status {
                font-size: 11px;
                opacity: 0.9;
            }
            .sc-omnia-messages {
                max-height: 350px;
                overflow-y: auto;
                padding: 12px 12px 8px;
                background: rgba(255, 255, 255, 0.94);
            }
            .sc-omnia-msg {
                margin-bottom: 8px;
                display: flex;
            }
            .sc-omnia-msg.sc-omnia-bot {
                justify-content: flex-start;
            }
            .sc-omnia-msg.sc-omnia-user {
                justify-content: flex-end;
            }
            .sc-omnia-bubble {
                max-width: 80%;
                padding: 8px 11px;
                border-radius: 16px;
                font-size: 13px;
                line-height: 1.7;
            }
            .sc-omnia-msg.sc-omnia-bot .sc-omnia-bubble {
                background: rgba(110, 69, 255, 0.08);
                border: 1px solid rgba(110, 69, 255, 0.2);
                color: #2b2345;
                border-bottom-left-radius: 4px;
            }
            .sc-omnia-msg.sc-omnia-user .sc-omnia-bubble {
                background: #6e45ff;
                color: #fff;
                border-bottom-right-radius: 4px;
            }
            .sc-omnia-input-row {
                display: flex;
                gap: 8px;
                padding: 10px 12px;
                border-top: 1px solid rgba(210, 195, 255, 0.7);
                background: rgba(249, 247, 255, 0.95);
            }
            .sc-omnia-input-row input {
                flex: 1;
                border-radius: 999px;
                border: 1px solid rgba(175, 160, 255, 0.7);
                padding: 8px 12px;
                font-size: 13px;
                outline: none;
                background: #fff;
            }
            .sc-omnia-input-row input:focus {
                border-color: #6e45ff;
                box-shadow: 0 0 0 1px rgba(110, 69, 255, 0.3);
            }
            .sc-omnia-input-row button {
                border-radius: 999px;
                border: none;
                padding: 8px 14px;
                font-size: 13px;
                font-weight: 600;
                cursor: pointer;
                background: linear-gradient(135deg, #6b46ff, #c77dff);
                color: #fff;
                box-shadow: 0 8px 18px rgba(110, 69, 255, 0.35);
            }
            .sc-omnia-input-row button:disabled {
                opacity: 0.6;
                box-shadow: none;
                cursor: default;
            }
        </style>

        <script>
            (function() {
                const form = document.getElementById('sc-omnia-form');
                const input = document.getElementById('sc-omnia-input');
                const messagesBox = document.getElementById('sc-omnia-messages');
                let isSending = false;

                function appendMessage(text, type) {
                    const msg = document.createElement('div');
                    msg.className = 'sc-omnia-msg ' + (type === 'user' ? 'sc-omnia-user' : 'sc-omnia-bot');
                    const bubble = document.createElement('div');
                    bubble.className = 'sc-omnia-bubble';
                    bubble.innerHTML = text;
                    msg.appendChild(bubble);
                    messagesBox.appendChild(msg);
                    messagesBox.scrollTop = messagesBox.scrollHeight;
                }

                form.addEventListener('submit', function(e) {
                    e.preventDefault();
                    if (isSending) return;

                    const value = (input.value || '').trim();
                    if (!value) return;

                    appendMessage(value, 'user');
                    input.value = '';
                    isSending = true;
                    form.querySelector('button').disabled = true;

                    const formData = new FormData();
                    formData.append('action', 'sitycloud_ai_omnia_chat');
                    formData.append('nonce', '<?php echo esc_js($nonce); ?>');
                    formData.append('message', value);

                    fetch('<?php echo esc_js(admin_url('admin-ajax.php')); ?>', {
                        method: 'POST',
                        body: formData,
                        credentials: 'same-origin'
                    })
                    .then(res => res.json())
                    .then(json => {
                        isSending = false;
                        form.querySelector('button').disabled = false;
                        if (json.success) {
                            appendMessage(json.data.message || '', 'bot');
                        } else {
                            appendMessage('حصل خطأ في الاتصال بالخدمة. جرّب تاني بعد شوية.', 'bot');
                        }
                    })
                    .catch(() => {
                        isSending = false;
                        form.querySelector('button').disabled = false;
                        appendMessage('تعذر الاتصال بالسيرفر. تأكد من الاتصال بالإنترنت.', 'bot');
                    });
                });
            })();
        </script>
        <?php
        return ob_get_clean();
    }

    // ==================================================
    //  Render Admin Page (UI + CSS + JS)
    // ==================================================
    public function render_admin_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        $nonce = wp_create_nonce('sitycloud_ai_nonce');

        ?>
        <div class="sitycloud-ai-wrapper">
            <div class="sitycloud-ai-header">
                <div>
                    <h1>SityCloud AI Control Center</h1>
                    <p>وكيل ذكاء اصطناعي مركزي لإدارة شبكة Sity Cloud: بحث، حماية، تقارير، ودعم عملاء "أمنية".</p>
                </div>
                <div class="sitycloud-ai-badge">
                    <span class="dot"></span>
                    Omnia • Gemini-powered Agent
                </div>
            </div>

            <div class="sitycloud-ai-layout">
                <!-- Sidebar Tabs -->
                <div class="sitycloud-ai-sidebar">
                    <button class="sitycloud-ai-tab active" data-target="tab-search">
                        🔎 بحث داخل شبكة Sity Cloud
                    </button>
                    <button class="sitycloud-ai-tab" data-target="tab-debugger">
                        🛠 AI Debugger
                    </button>
                    <button class="sitycloud-ai-tab" data-target="tab-support">
                        💬 أمنية – دعم العملاء (لوحة التحكم)
                    </button>
                    <button class="sitycloud-ai-tab" data-target="tab-auto-fix">
                        ⚡ Auto Fix الذكي
                    </button>
                    <button class="sitycloud-ai-tab" data-target="tab-security-agent">
                        🛡 وكيل الحماية – Security Agent
                    </button>
                    <button class="sitycloud-ai-tab" data-target="tab-report">
                        📊 تقرير إداري ذكي
                    </button>
                </div>

                <!-- Content Area -->
                <div class="sitycloud-ai-content">
                    <!-- TAB: Search Network -->
                    <div id="tab-search" class="sitycloud-ai-tab-content active">
                        <h2>بحث ذكي داخل شبكة Sity Cloud</h2>
                        <p class="tab-desc">
                            ابحث عن مواقع عملاء Sity Cloud حسب النشاط، الاسم، أو استخدام معين، والوكيل يحلل الشبكة ويعرض الأنسب.
                        </p>
                        <div class="sitycloud-card glass">
                            <label class="sitycloud-label">استعلام البحث</label>
                            <textarea id="sitycloud-search-query" class="sitycloud-input" rows="3" placeholder="مثال: مواقع تعليمية، مواقع مطاعم، مواقع شركات ناشئة..."></textarea>
                            <button id="sitycloud-search-btn" class="sitycloud-btn primary">ابحث في شبكة Sity Cloud</button>
                        </div>
                        <div id="sitycloud-search-result" class="sitycloud-result-box"></div>
                    </div>

                    <!-- TAB: AI Debugger -->
                    <div id="tab-debugger" class="sitycloud-ai-tab-content">
                        <h2>AI Debugger لمواقع Sity Cloud</h2>
                        <p class="tab-desc">
                            فحص ذكي لإعدادات الموقع، الإضافات، والثيم، مع تحليل احترافي واقتراحات لتحسين الاستقرار والأداء.
                        </p>
                        <div class="sitycloud-card glass">
                            <p>اضغط الزر ليتم جمع معلومات تقنية عن موقع Sity Cloud الحالي وتحليلها من خلال الوكيل الذكي.</p>
                            <button id="sitycloud-debugger-btn" class="sitycloud-btn primary">تشغيل فحص Debug</button>
                        </div>
                        <div id="sitycloud-debugger-result" class="sitycloud-result-box"></div>
                    </div>

                    <!-- TAB: Omnia Support (Admin) -->
                    <div id="tab-support" class="sitycloud-ai-tab-content">
                        <h2>أمنية – وكيل دعم عملاء Sity Cloud</h2>
                        <p class="tab-desc">
                            جرّب رد أمنية على تذاكر العملاء قبل إرسالها أو لاستخدام الرد كنقطة بداية في نظام التذاكر الخاص بك.
                        </p>
                        <div class="sitycloud-grid-2">
                            <div class="sitycloud-card glass">
                                <label class="sitycloud-label">وصف مشكلة / استفسار العميل</label>
                                <textarea id="sitycloud-support-message" class="sitycloud-input" rows="6" placeholder="اكتب هنا نص الرسالة القادمة من العميل أو ملخّص المشكلة..."></textarea>
                                <button id="sitycloud-support-btn" class="sitycloud-btn primary">توليد رد أمنية</button>
                            </div>
                            <div class="sitycloud-card glass">
                                <h3>رد أمنية المقترح</h3>
                                <div id="sitycloud-support-result" class="sitycloud-result-box small"></div>
                            </div>
                        </div>
                    </div>

                    <!-- TAB: AI Auto Fix -->
                    <div id="tab-auto-fix" class="sitycloud-ai-tab-content">
                        <h2>Auto Fix الذكي (آمن)</h2>
                        <p class="tab-desc">
                            تنفيذ إصلاحات آمنة مثل تحديث Permalinks مع تقرير توضيحي من الوكيل الذكي حول تحسين SSL و PageSpeed.
                        </p>
                        <div class="sitycloud-card glass">
                            <p>لن يتم عمل أي تعديل خطير. يتم فقط تشغيل خطوات آمنة داخل منصة Sity Cloud، مع تقرير يوضح ما تم واقتراحات إضافية.</p>
                            <button id="sitycloud-auto-fix-btn" class="sitycloud-btn danger">تشغيل Auto Fix الآمن بموافقتك</button>
                        </div>
                        <div id="sitycloud-auto-fix-result" class="sitycloud-result-box"></div>
                    </div>

                    <!-- TAB: Security Agent -->
                    <div id="tab-security-agent" class="sitycloud-ai-tab-content">
                        <h2>وكيل الحماية – Security Agent</h2>
                        <p class="tab-desc">
                            وكيل حماية يقوم بتنفيذ مهام أمان أساسية داخل موقع Sity Cloud بعد موافقتك، مع تقرير واضح بما تم.
                        </p>
                        <div class="sitycloud-card glass">
                            <p style="margin-bottom:8px;">الإجراءات المقترحة (آمنة ويمكن التراجع عنها لاحقًا):</p>
                            <ul class="sitycloud-security-list">
                                <li>إيقاف تسجيل مستخدمين جدد من الواجهة العامة (Users can register).</li>
                                <li>تشغيل مراجعة التعليقات قبل النشر (Comment moderation).</li>
                                <li>منع الموافقة التلقائية للتعليقات الجديدة.</li>
                            </ul>
                            <p>لن يتم تنفيذ أي خطوة إلا بعد ضغطك على الزر أدناه.</p>
                            <button id="sitycloud-security-btn" class="sitycloud-btn primary">تنفيذ مهام الحماية الآن</button>
                        </div>
                        <div id="sitycloud-security-result" class="sitycloud-result-box"></div>
                    </div>

                    <!-- TAB: AI Report Generator -->
                    <div id="tab-report" class="sitycloud-ai-tab-content">
                        <h2>تقرير إداري ذكي لموقع Sity Cloud</h2>
                        <p class="tab-desc">
                            تقرير يشمل السرعة المتوقعة، الأمان، الإضافات، والأخطاء المحتملة – بصياغة واضحة يمكنك مشاركتها مع العميل.
                        </p>
                        <div class="sitycloud-card glass">
                            <p>اضغط توليد تقرير للحصول على رؤية شاملة عن حالة موقع Sity Cloud الحالي.</p>
                            <button id="sitycloud-report-btn" class="sitycloud-btn primary">توليد التقرير الآن</button>
                        </div>
                        <div id="sitycloud-report-result" class="sitycloud-result-box"></div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Nonce & AJAX URL -->
        <script>
            window.SityCloudAI = {
                ajaxUrl: "<?php echo esc_js(admin_url('admin-ajax.php')); ?>",
                nonce: "<?php echo esc_js($nonce); ?>"
            };
        </script>

        <!-- Inline CSS (Glassmorphism / White / Light Purple) -->
        <style>
            .sitycloud-ai-wrapper {
                font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
                direction: rtl;
                text-align: right;
                padding: 24px;
                background:
                    radial-gradient(circle at top left, rgba(120, 90, 255, 0.09) 0, transparent 42%),
                    radial-gradient(circle at bottom right, rgba(193, 125, 255, 0.12) 0, transparent 50%),
                    #f8f7ff;
                min-height: calc(100vh - 32px);
            }
            .sitycloud-ai-header {
                display: flex;
                align-items: center;
                justify-content: space-between;
                gap: 16px;
                margin-bottom: 24px;
            }
            .sitycloud-ai-header h1 {
                margin: 0;
                font-size: 28px;
                font-weight: 800;
                letter-spacing: 0.03em;
                background: linear-gradient(135deg, #5f3df7, #c17dff);
                -webkit-background-clip: text;
                color: transparent;
            }
            .sitycloud-ai-header p {
                margin: 4px 0 0;
                color: #555;
                font-size: 14px;
            }
            .sitycloud-ai-badge {
                display: inline-flex;
                align-items: center;
                gap: 8px;
                padding: 9px 16px;
                border-radius: 999px;
                background: rgba(255, 255, 255, 0.85);
                box-shadow: 0 10px 30px rgba(120, 90, 255, 0.2);
                border: 1px solid rgba(145, 122, 255, 0.35);
                font-size: 12px;
                color: #4a3cb1;
                backdrop-filter: blur(16px);
            }
            .sitycloud-ai-badge .dot {
                width: 8px;
                height: 8px;
                border-radius: 50%;
                background: #4ade80;
                box-shadow: 0 0 8px rgba(74, 222, 128, 0.8);
            }
            .sitycloud-ai-layout {
                display: grid;
                grid-template-columns: 260px minmax(0, 1fr);
                gap: 20px;
            }
            .sitycloud-ai-sidebar {
                background: rgba(255, 255, 255, 0.78);
                border-radius: 22px;
                padding: 12px;
                box-shadow: 0 20px 45px rgba(100, 80, 220, 0.16);
                border: 1px solid rgba(185, 170, 255, 0.35);
                backdrop-filter: blur(18px);
                display: flex;
                flex-direction: column;
                gap: 8px;
            }
            .sitycloud-ai-tab {
                border: none;
                outline: none;
                text-align: right;
                padding: 10px 12px;
                border-radius: 999px;
                background: transparent;
                color: #4b4f5c;
                font-size: 13px;
                cursor: pointer;
                display: flex;
                justify-content: flex-start;
                align-items: center;
                gap: 6px;
                transition: all 0.2s ease;
            }
            .sitycloud-ai-tab.active {
                background: linear-gradient(135deg, #6e45ff, #c17dff);
                color: #fff;
                box-shadow: 0 12px 30px rgba(110, 69, 255, 0.35);
                transform: translateY(-1px);
            }
            .sitycloud-ai-tab:hover:not(.active) {
                background: rgba(111, 90, 255, 0.06);
            }

            .sitycloud-ai-content {
                background: rgba(255, 255, 255, 0.95);
                border-radius: 26px;
                padding: 20px 22px;
                box-shadow: 0 20px 50px rgba(100, 80, 220, 0.18);
                border: 1px solid rgba(200, 190, 255, 0.5);
                backdrop-filter: blur(22px);
            }
            .sitycloud-ai-tab-content {
                display: none;
            }
            .sitycloud-ai-tab-content.active {
                display: block;
            }
            .sitycloud-ai-tab-content h2 {
                margin-top: 0;
                margin-bottom: 6px;
                font-size: 20px;
                color: #232440;
            }
            .tab-desc {
                margin-top: 0;
                margin-bottom: 16px;
                font-size: 13px;
                color: #6b6f82;
            }

            .sitycloud-card {
                border-radius: 18px;
                padding: 16px 18px;
                margin-bottom: 16px;
            }
            .sitycloud-card.glass {
                background: rgba(255, 255, 255, 0.88);
                border: 1px solid rgba(187, 170, 255, 0.6);
                box-shadow: 0 16px 35px rgba(103, 84, 235, 0.17);
                backdrop-filter: blur(16px);
            }

            .sitycloud-label {
                font-size: 13px;
                font-weight: 600;
                margin-bottom: 6px;
                display: inline-block;
                color: #3b3659;
            }
            .sitycloud-input {
                width: 100%;
                border-radius: 14px;
                border: 1px solid rgba(150, 135, 255, 0.5);
                padding: 10px 12px;
                font-size: 13px;
                resize: vertical;
                background: rgba(255, 255, 255, 0.96);
                outline: none;
                transition: box-shadow 0.2s ease, border-color 0.2s ease, background 0.2s ease;
            }
            .sitycloud-input:focus {
                border-color: #7b5cff;
                box-shadow: 0 0 0 1px rgba(123, 92, 255, 0.35);
                background: #ffffff;
            }

            .sitycloud-btn {
                display: inline-flex;
                align-items: center;
                justify-content: center;
                gap: 6px;
                margin-top: 10px;
                padding: 9px 16px;
                border-radius: 999px;
                border: none;
                cursor: pointer;
                font-size: 13px;
                font-weight: 600;
                letter-spacing: 0.02em;
                transition: all 0.2s ease;
            }
            .sitycloud-btn.primary {
                background: linear-gradient(135deg, #6e45ff, #c17dff);
                color: #ffffff;
                box-shadow: 0 12px 30px rgba(110, 69, 255, 0.4);
            }
            .sitycloud-btn.primary:hover {
                transform: translateY(-1px) scale(1.01);
                box-shadow: 0 16px 40px rgba(110, 69, 255, 0.5);
            }
            .sitycloud-btn.danger {
                background: linear-gradient(135deg, #ff4b8a, #ff7a6e);
                color: #ffffff;
                box-shadow: 0 12px 30px rgba(255, 75, 138, 0.35);
            }
            .sitycloud-btn.danger:hover {
                transform: translateY(-1px) scale(1.01);
                box-shadow: 0 16px 40px rgba(255, 75, 138, 0.5);
            }

            .sitycloud-result-box {
                margin-top: 12px;
                padding: 12px 14px;
                border-radius: 14px;
                background: rgba(247, 245, 255, 0.9);
                border: 1px solid rgba(200, 190, 255, 0.7);
                font-size: 13px;
                color: #333447;
                max-height: 450px;
                overflow: auto;
                white-space: normal;
            }
            .sitycloud-result-box.small {
                max-height: 260px;
            }
            .sitycloud-result-box.loading::before {
                content: "⏳ جاري معالجة طلبك بواسطة وكيل Sity Cloud الذكي...";
                display: block;
                margin-bottom: 6px;
                color: #5b4be0;
                font-weight: 600;
            }

            .sitycloud-grid-2 {
                display: grid;
                grid-template-columns: minmax(0, 1.1fr) minmax(0, 1fr);
                gap: 16px;
            }

            .sitycloud-security-list {
                margin: 0 0 8px;
                padding-right: 18px;
                font-size: 13px;
                color: #3f3f55;
            }
            .sitycloud-security-list li {
                margin-bottom: 4px;
            }

            @media (max-width: 960px) {
                .sitycloud-ai-layout {
                    grid-template-columns: 1fr;
                }
                .sitycloud-grid-2 {
                    grid-template-columns: 1fr;
                }
            }
        </style>

        <!-- Inline JS -->
        <script>
            (function() {
                const tabs = document.querySelectorAll('.sitycloud-ai-tab');
                const contents = document.querySelectorAll('.sitycloud-ai-tab-content');

                tabs.forEach(tab => {
                    tab.addEventListener('click', () => {
                        tabs.forEach(t => t.classList.remove('active'));
                        contents.forEach(c => c.classList.remove('active'));

                        tab.classList.add('active');
                        const target = document.getElementById(tab.dataset.target);
                        if (target) {
                            target.classList.add('active');
                        }
                    });
                });

                function ajaxPost(action, data, resultElementId) {
                    const resultBox = document.getElementById(resultElementId);
                    if (!resultBox) return;

                    resultBox.classList.add('loading');
                    resultBox.innerHTML = '';

                    const formData = new FormData();
                    formData.append('action', action);
                    formData.append('_ajax_nonce', window.SityCloudAI.nonce);

                    for (const key in data) {
                        formData.append(key, data[key]);
                    }

                    fetch(window.SityCloudAI.ajaxUrl, {
                        method: 'POST',
                        credentials: 'same-origin',
                        body: formData
                    })
                    .then(res => res.json())
                    .then(json => {
                        resultBox.classList.remove('loading');

                        if (json.success) {
                            resultBox.innerHTML = json.data.message || '';
                        } else {
                            resultBox.innerHTML = 'حدث خطأ: ' + (json.data && json.data.message ? json.data.message : 'خطأ غير معروف.');
                        }
                    })
                    .catch(err => {
                        resultBox.classList.remove('loading');
                        resultBox.innerHTML = 'تعذر الاتصال بوكلاء Sity Cloud. تأكد من الاتصال ومفتاح الـ API.';
                    });
                }

                // Search Network
                const searchBtn = document.getElementById('sitycloud-search-btn');
                if (searchBtn) {
                    searchBtn.addEventListener('click', () => {
                        const query = document.getElementById('sitycloud-search-query').value || '';
                        ajaxPost('sitycloud_ai_search_network', { query: query }, 'sitycloud-search-result');
                    });
                }

                // Debugger
                const debugBtn = document.getElementById('sitycloud-debugger-btn');
                if (debugBtn) {
                    debugBtn.addEventListener('click', () => {
                        ajaxPost('sitycloud_ai_debugger', {}, 'sitycloud-debugger-result');
                    });
                }

                // Support Agent (Omnia – Admin)
                const supportBtn = document.getElementById('sitycloud-support-btn');
                if (supportBtn) {
                    supportBtn.addEventListener('click', () => {
                        const msg = document.getElementById('sitycloud-support-message').value || '';
                        ajaxPost('sitycloud_ai_support', { message: msg }, 'sitycloud-support-result');
                    });
                }

                // Auto Fix
                const autoFixBtn = document.getElementById('sitycloud-auto-fix-btn');
                if (autoFixBtn) {
                    autoFixBtn.addEventListener('click', () => {
                        if (!confirm('سيتم تشغيل Auto Fix الآمن داخل Sity Cloud. متأكد من الموافقة؟')) return;
                        ajaxPost('sitycloud_ai_auto_fix', {}, 'sitycloud-auto-fix-result');
                    });
                }

                // Security Agent
                const securityBtn = document.getElementById('sitycloud-security-btn');
                if (securityBtn) {
                    securityBtn.addEventListener('click', () => {
                        if (!confirm('سيتم تنفيذ مهام حماية آمنة داخل موقع Sity Cloud الحالي. هل توافق؟')) return;
                        ajaxPost('sitycloud_ai_security_agent', {}, 'sitycloud-security-result');
                    });
                }

                // Report
                const reportBtn = document.getElementById('sitycloud-report-btn');
                if (reportBtn) {
                    reportBtn.addEventListener('click', () => {
                        ajaxPost('sitycloud_ai_report', {}, 'sitycloud-report-result');
                    });
                }
            })();
        </script>
        <?php
    }

    // ==================================================
    //  Helper: Call Gemini API (2.0 Flash)
// ==================================================
    private function call_gemini_api($prompt) {
        if (!SITYCLOUD_GEMINI_API_KEY || SITYCLOUD_GEMINI_API_KEY === 'PUT_YOUR_GEMINI_API_KEY_HERE') {
            return new WP_Error('no_api_key', 'مفتاح Gemini API غير مضبوط في البلجن.');
        }

        // الموديل الجديد
        $endpoint = 'https://generativelanguage.googleapis.com/v1beta/models/gemini-2.0-flash:generateContent';

        // جسم الطلب
        $body = array(
            'contents' => array(
                array(
                    'parts' => array(
                        array('text' => $prompt)
                    )
                )
            )
        );

        // إرسال الطلب
        $response = wp_remote_post($endpoint, array(
            'headers' => array(
                'Content-Type'   => 'application/json',
                'X-goog-api-key' => SITYCLOUD_GEMINI_API_KEY,
            ),
            'body'    => wp_json_encode($body),
            'timeout' => 30
        ));

        // أخطاء اتصال
        if (is_wp_error($response)) {
            return new WP_Error('http_error', 'تعذر الاتصال بخدمة Gemini: ' . $response->get_error_message());
        }

        $code = wp_remote_retrieve_response_code($response);
        $raw  = wp_remote_retrieve_body($response);
        $data = json_decode($raw, true);

        // لو Google رجعت Error
        if (isset($data['error'])) {
            $msg = isset($data['error']['message']) ? $data['error']['message'] : 'خطأ من خدمة Gemini.';
            return new WP_Error('api_error', 'خطأ من Gemini: ' . $msg);
        }

        // التأكد من وجود نص
        if ($code !== 200 || empty($data['candidates'][0]['content']['parts'][0]['text'])) {
            return new WP_Error('bad_response', 'رد غير متوقع من Gemini: ' . substr($raw, 0, 400));
        }

        return $data['candidates'][0]['content']['parts'][0]['text'];
    }

    // ==================================================
    //  Helper: sanitize AI text (remove #, * and format)
// ==================================================
    private function format_ai_message($text, $prefix_html = '') {
        // إزالة علامات Markdown المزعجة
        $clean = str_replace(array('#', '*'), '', $text);
        // تأمين النص
        $clean = esc_html($clean);
        // تحويل السطور الجديدة لـ <br>
        $clean = nl2br($clean);
        return $prefix_html . $clean;
    }

    private function check_ajax_permissions() {
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'غير مسموح لك بتنفيذ هذا الإجراء.'), 403);
        }
        check_ajax_referer('sitycloud_ai_nonce');
    }

    // ==================================================
    //  AJAX: Search Network
    // ==================================================
    public function ajax_search_network() {
        $this->check_ajax_permissions();

        $query = isset($_POST['query']) ? sanitize_text_field(wp_unslash($_POST['query'])) : '';

        $sites_summary = '';

        if (is_multisite()) {
            $sites = get_sites(array(
                'number' => 1000,
            ));

            $lines = array();
            foreach ($sites as $site) {
                $details = get_blog_details($site->blog_id);
                $lines[] = sprintf(
                    "Site #%d: Name: %s | URL: %s",
                    $site->blog_id,
                    $details && $details->blogname ? $details->blogname : '(no name)',
                    $details && $details->siteurl ? $details->siteurl : $site->domain . $site->path
                );
            }
            $sites_summary = implode("\n", $lines);
        } else {
            $sites_summary = "Single Sity Cloud site only: " . home_url('/') . " | " . get_bloginfo('name');
        }

        $prompt =
            "You are an AI assistant (agent) helping an admin manage a Sity Cloud network (built on WordPress but you must NOT say the word 'WordPress' in Arabic, use 'منصة Sity Cloud' instead).\n\n" .
            "User query (Arabic):\n" . $query . "\n\n" .
            "Below is the list of Sity Cloud sites in the network:\n" . $sites_summary . "\n\n" .
            "Answer in professional Arabic about Sity Cloud only. Analyze which sites match the query best and present them في شكل نقاط واضحة بدون استخدام رموز Markdown أو علامات مثل # أو *.\n" .
            "اكتب:\n- اسم الموقع\n- الرابط\n- سبب الاختيار\nبدون استخدام أي ترقيم Markdown، فقط نص عادي.";

        $response = $this->call_gemini_api($prompt);

        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => $response->get_error_message()));
        }

        $html = $this->format_ai_message($response);
        wp_send_json_success(array('message' => $html));
    }

    // ==================================================
    //  AJAX: AI Debugger
    // ==================================================
    public function ajax_ai_debugger() {
        $this->check_ajax_permissions();

        global $wpdb;

        $debug_data = array(
            'site_url'   => site_url(),
            'home_url'   => home_url(),
            'site_name'  => get_bloginfo('name'),
            'platform'   => 'Sity Cloud site (multisite possible)',
            'wp_version' => get_bloginfo('version'),
            'php_version'=> phpversion(),
            'is_multisite' => is_multisite() ? 'yes' : 'no',
            'active_theme' => wp_get_theme()->get('Name'),
            'template'     => wp_get_theme()->get_template(),
            'wp_debug'     => defined('WP_DEBUG') ? (WP_DEBUG ? 'true' : 'false') : 'not defined',
            'db_version'   => $wpdb->db_version(),
        );

        // Active plugins
        $active_plugins = (array) get_option('active_plugins', array());
        if (is_multisite()) {
            $network_plugins = array_keys((array) get_site_option('active_sitewide_plugins', array()));
            $active_plugins = array_unique(array_merge($active_plugins, $network_plugins));
        }

        if (!function_exists('get_plugin_data')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $plugins_list = array();
        foreach ($active_plugins as $plugin_file) {
            $file_path = WP_PLUGIN_DIR . '/' . $plugin_file;
            if (file_exists($file_path)) {
                $data = get_plugin_data($file_path, false, false);
                $plugins_list[] = $data['Name'] . ' (v' . $data['Version'] . ')';
            }
        }

        $prompt =
            "You are an advanced debugging assistant for a site running on Sity Cloud (do not mention 'WordPress' explicitly in Arabic, always say منصة Sity Cloud).\n\n" .
            "Here is the technical info:\n" . print_r($debug_data, true) . "\n\n" .
            "Active plugins:\n" . implode("\n", $plugins_list) . "\n\n" .
            "Return the answer in clear, well-structured Arabic but as normal text (no Markdown symbols, no #, no *).\n" .
            "قسم الإجابة إلى فقرات أساسية:\n" .
            "ملخص عام عن حالة موقع Sity Cloud.\n" .
            "مشاكل محتملة في الأداء أو التوافق أو الأمان.\n" .
            "توصيات تقنية واضحة.\n" .
            "نقاط ذات أولوية عالية للتنفيذ.\n" .
            "استخدم فواصل أسطر عادية فقط.";

        $response = $this->call_gemini_api($prompt);

        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => $response->get_error_message()));
        }

        $html = $this->format_ai_message($response);
        wp_send_json_success(array('message' => $html));
    }

    // ==================================================
    //  AJAX: AI Support Agent (Omnia – Admin)
// ==================================================
    public function ajax_ai_support() {
        $this->check_ajax_permissions();

        $message = isset($_POST['message']) ? wp_kses_post(wp_unslash($_POST['message'])) : '';
        if (empty($message)) {
            wp_send_json_error(array('message' => 'من فضلك اكتب محتوى رسالة العميل أولاً.'));
        }

        $prompt =
            "You are 'Omnia', an AI customer support agent for Sity Cloud. You speak Arabic only. " .
            "You must not mention the word 'WordPress' in Arabic; always say منصة Sity Cloud instead.\n\n" .
            "Customer message (Arabic):\n" . $message . "\n\n" .
            "Respond in warm, professional Arabic. No Markdown, no #, no *.\n" .
            "الرد يكون:\n" .
            "ترحيب بسيط.\n" .
            "تلخيص المشكلة أو السؤال.\n" .
            "خطوات واضحة لحل المشكلة أو توضيح المطلوب.\n" .
            "لو في حاجة تتصعد لفريق بشري، وضح ذلك في نهاية الرد.\n" .
            "اكتب النص كسطور عادية فقط بدون رموز غريبة.";

        $response = $this->call_gemini_api($prompt);

        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => $response->get_error_message()));
        }

        $html = $this->format_ai_message($response);
        wp_send_json_success(array('message' => $html));
    }

    // ==================================================
    //  AJAX: AI Auto Fix (safe actions + AI explanation)
// ==================================================
    public function ajax_ai_auto_fix() {
        $this->check_ajax_permissions();

        // Safe Auto Fix: flush rewrite rules (Permalinks)
        flush_rewrite_rules();

        // جمع شوية بيانات لشرح ما تم
        $data = array(
            'site_url' => site_url(),
            'home_url' => home_url(),
            'permalink_structure' => get_option('permalink_structure'),
            'is_ssl' => is_ssl() ? 'yes' : 'no',
        );

        $prompt =
            "You are an AI assistant agent for Sity Cloud running a safe Auto Fix routine. " .
            "The platform is Sity Cloud. Do not say 'WordPress' in Arabic.\n\n" .
            "We already executed some safe automatic fixes in Sity Cloud (like flushing rewrite rules for permalinks).\n" .
            "Context:\n" . print_r($data, true) . "\n\n" .
            "Explain in Arabic (no Markdown, no #, no *):\n" .
            "ما الذي تم إصلاحه بشكل آمن.\n" .
            "نصائح لتحسين SSL في منصة Sity Cloud.\n" .
            "أفكار لتحسين سرعة الموقع وتجربة المستخدم.\n" .
            "توصيات عامة لاستقرار وأمان أفضل.\n" .
            "اكتب فقرات بنقاط عادية وسطور جديدة فقط بدون أي ترميز Markdown.";

        $response = $this->call_gemini_api($prompt);

        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => $response->get_error_message()));
        }

        $ai_html = $this->format_ai_message($response, '<strong>تم تنفيذ Auto Fix الآمن داخل Sity Cloud.</strong><br><br>');
        wp_send_json_success(array('message' => $ai_html));
    }

    // ==================================================
    //  AJAX: Security Agent – Safe Protection Tasks
    // ==================================================
    public function ajax_ai_security_agent() {
        $this->check_ajax_permissions();

        // تنفيذ مهام حماية بسيطة وآمنة بموافقة الأدمن:
        // 1) إيقاف تسجيل مستخدمين جدد من الواجهة
        update_option('users_can_register', 0);

        // 2) تفعيل مراجعة التعليقات قبل النشر
        update_option('comment_moderation', 1);
        update_option('comment_previously_approved', 0);

        $security_state = array(
            'users_can_register'          => get_option('users_can_register'),
            'comment_moderation'          => get_option('comment_moderation'),
            'comment_previously_approved' => get_option('comment_previously_approved'),
            'site_url'                    => site_url(),
        );

        $prompt =
            "You are a security agent for Sity Cloud. The admin approved executing basic protection actions.\n\n" .
            "We have just applied the following safe changes on this Sity Cloud site:\n" .
            "- Disabled public user registration.\n" .
            "- Enabled comment moderation.\n" .
            "- Disabled auto-approval for new comments.\n\n" .
            "Here is the state:\n" . print_r($security_state, true) . "\n\n" .
            "Explain in Arabic (no Markdown, no #, no *):\n" .
            "ما الذي تم تنفيذه تحديداً لحماية الموقع.\n" .
            "كيف تساعد هذه التغييرات في تقليل السبام وتقوية الأمان.\n" .
            "نصائح إضافية بسيطة يمكن للإدمن تنفيذها داخل منصة Sity Cloud.\n" .
            "اكتب النص في شكل فقرات وسطور واضحة بدون أي رموز Markdown.";

        $response = $this->call_gemini_api($prompt);
        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => $response->get_error_message()));
        }

        $html = $this->format_ai_message($response, '<strong>تم تنفيذ مهام الحماية الأساسية داخل Sity Cloud.</strong><br><br>');
        wp_send_json_success(array('message' => $html));
    }

    // ==================================================
    //  AJAX: AI Report Generator
    // ==================================================
    public function ajax_ai_report() {
        $this->check_ajax_permissions();

        global $wpdb;

        // Basic site info
        $theme = wp_get_theme();
        $info = array(
            'site_url'      => site_url(),
            'home_url'      => home_url(),
            'site_name'     => get_bloginfo('name'),
            'platform_name' => 'Sity Cloud (do not say WordPress explicitly in Arabic)',
            'wp_version'    => get_bloginfo('version'),
            'php_version'   => phpversion(),
            'theme_name'    => $theme->get('Name'),
            'theme_version' => $theme->get('Version'),
            'is_multisite'  => is_multisite() ? 'yes' : 'no',
        );

        // Plugins
        $active_plugins = (array) get_option('active_plugins', array());
        if (is_multisite()) {
            $network_plugins = array_keys((array) get_site_option('active_sitewide_plugins', array()));
            $active_plugins = array_unique(array_merge($active_plugins, $network_plugins));
        }

        if (!function_exists('get_plugin_data')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $plugins_list = array();
        foreach ($active_plugins as $plugin_file) {
            $file_path = WP_PLUGIN_DIR . '/' . $plugin_file;
            if (file_exists($file_path)) {
                $data = get_plugin_data($file_path, false, false);
                $plugins_list[] = $data['Name'] . ' (v' . $data['Version'] . ')';
            }
        }

        // Logs info (basic)
        $log_hint = '';
        if (defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
            $log_hint = 'WP_DEBUG_LOG is enabled. Logs are usually in wp-content/debug.log';
        } else {
            $log_hint = 'WP_DEBUG_LOG is disabled. Enable it in wp-config.php to capture detailed PHP logs.';
        }

        $prompt =
            "You are an AI Report Generator for Sity Cloud.\n\n" .
            "Goal: produce a clean, executive-style report in Arabic for the admin covering:\n" .
            "- سرعة الموقع المتوقعة.\n" .
            "- الأمان.\n" .
            "- الإضافات.\n" .
            "- الأخطاء المحتملة.\n" .
            "- ملاحظات عن سجلات PHP.\n\n" .
            "Do not mention the word 'WordPress' in Arabic. Say منصة Sity Cloud instead.\n\n" .
            "Site info:\n" . print_r($info, true) . "\n\n" .
            "Active plugins:\n" . implode("\n", $plugins_list) . "\n\n" .
            "Logs hint:\n" . $log_hint . "\n\n" .
            "You do NOT have real PageSpeed numbers, so speak in terms of تقدير واحتمالات.\n\n" .
            "Return the report in Arabic as normal text, no Markdown symbols, no #, no *.\n" .
            "قسّم التقرير إلى فقرات بعناوين نصية بسيطة وسطور جديدة فقط يمكن للإدمن إرسالها للعميل.";

        $response = $this->call_gemini_api($prompt);

        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => $response->get_error_message()));
        }

        $html = $this->format_ai_message($response);
        wp_send_json_success(array('message' => $html));
    }

    // ==================================================
    //  AJAX: Omnia Chat (Frontend chatbot)
// ==================================================
    public function ajax_ai_omnia_chat() {
        // هنا ماينفعش نطلب manage_options لأن ده للعميل
        check_ajax_referer('sitycloud_ai_omnia', 'nonce');

        $message = isset($_POST['message']) ? sanitize_text_field(wp_unslash($_POST['message'])) : '';
        if (empty($message)) {
            wp_send_json_error(array('message' => 'الرسالة فاضية.'));
        }

        $prompt =
            "You are 'Omnia', an AI support agent for Sity Cloud. You chat with end customers in Arabic only.\n\n" .
            "Guidelines:\n" .
            "- Be friendly and clear.\n" .
            "- Explain Sity Cloud as a platform to build and manage websites بسهولة.\n" .
            "- Do not say the word 'WordPress' in Arabic; say منصة Sity Cloud.\n" .
            "- No Markdown, no #, no *, no bullet syntax. Just normal text.\n\n" .
            "Customer message (Arabic):\n" . $message . "\n\n" .
            "Answer in Arabic as if you are أمنية من دعم عملاء Sity Cloud.\n" .
            "اكتب الرد في شكل جمل واضحة وسطور قصيرة بدون أي رموز غريبة.";

        $response = $this->call_gemini_api($prompt);
        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => $response->get_error_message()));
        }

        $html = $this->format_ai_message($response);
        wp_send_json_success(array('message' => $html));
    }
}

// Init plugin
new SityCloud_AI_Assistant();
