<?php
/**
 * REST API Kadence Starter Templates Rest.
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
/**
 * REST API Starter controller class.
 */
class Kadence_Cloud_Rest_Controller extends WP_REST_Controller {

	/**
	 * Include property name.
	 */
	const PROP_KEY = 'key';

	/**
	 * Include property name.
	 */
	const PROP_SITE = 'site';

	/**
	 * Include property name.
	 */
	const PROP_ID = 'id';

	/**
	 * Include property name.
	 */
	const PROP_TYPE = 'type';

	/**
	 * Include property name.
	 */
	const PROP_STYLE = 'style';

	/**
	 * Include property name.
	 */
	const PROP_RETURN_COUNT = 'items';

	/**
	 * Include property name.
	 */
	const PROP_RETURN_PAGE = 'page';


	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->namespace   = 'kadence-cloud/v1';
		$this->rest_base   = 'get';
		$this->info_base   = 'info';
		$this->pages_base  = 'pages';
		$this->single_base = 'single';
		$this->category_base = 'categories';
	}

	/**
	 * Registers the routes for the objects of the controller.
	 *
	 * @see register_rest_route()
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => '__return_true',
					'args'                => $this->get_collection_params(),
				),
			)
		);
		register_rest_route(
			$this->namespace,
			'/' . $this->info_base,
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_info' ),
					'permission_callback' => '__return_true',
					'args'                => $this->get_collection_params(),
				),
			)
		);
		register_rest_route(
			$this->namespace,
			'/' . $this->category_base,
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_categories' ),
					'permission_callback' => '__return_true',
					'args'                => $this->get_collection_params(),
				),
			)
		);
		register_rest_route(
			$this->namespace,
			'/' . $this->single_base,
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_single' ),
					'permission_callback' => '__return_true',
					'args'                => $this->get_analytics_params(),
				),
			)
		);
	}

	/**
	 * Retrieves a collection of objects.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
	 */
	public function get_single( $request ) {
		if ( $this->check_access( $request ) ) {
			$settings   = json_decode( get_option( 'kadence_cloud' ), true );
			$pattern_id = $request->get_param( self::PROP_ID );
			if ( empty( $pattern_id ) ) {
				return wp_send_json( __( 'Invalid Request, Incorrect Pattern', 'kadence-cloud' ) );
			}
			if ( isset( $settings['enable_analytics'] ) && $settings['enable_analytics'] ) {
				$pattern_type  = $request->get_param( self::PROP_TYPE );
				$pattern_style = $request->get_param( self::PROP_STYLE );
				$key           = $request->get_param( self::PROP_KEY );
				if ( empty( $pattern_id ) || empty( $pattern_type ) ) {
					return wp_send_json( __( 'Invalid Request, Incorrect Data', 'kadence-cloud' ) );
				}
				$data = array(
					'type'    => $pattern_type,
					'post_id' => absint( $pattern_id ),
					'style'   => ! empty( $pattern_style ) ? $pattern_style : 'light',
				);
				KadenceWP\KadenceCloud\Analytics_Dashboard_Util::record_event( $data );
			}
			$pattern = $this->get_pattern( $pattern_id );
			return wp_send_json( $pattern );

		} else {
			return wp_send_json( __( 'Invalid Request, Incorrect Access Key', 'kadence-cloud' ) );
		}
	}
	/**
	 * Get Pattern content from pattern id.
	 * 
	 * @param string $pattern_id the pattern id.
	 */
	public function get_pattern( $pattern_id ) {
		$pattern = get_post( $pattern_id );
		if ( ! empty( $pattern ) ) {
			$pattern_content = $pattern->post_content;
			if ( ! empty( $pattern_content ) ) {
				return $pattern_content;
			}
		}
		return wp_send_json( __( 'Invalid Request, Incorrect Pattern ID', 'kadence-cloud' ) );
	}
	/**
	 * Retrieves a collection of objects.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
	 */
	public function get_categories( $request ) {
		if ( $this->check_access( $request ) ) {
			$terms = get_terms(
				array(
					'taxonomy'   => 'kadence-cloud-categories',
					'hide_empty' => false,
					'orderby' => 'menu_order',
					'order' => 'ASC',
				)
			);
			$terms_array = array();
			if ( ! empty( $terms ) ) {
				foreach ( $terms as $key => $value ) {
					$terms_array[ $value->slug ] = $value->name;
				}
			}
			return rest_ensure_response( $terms_array );
		} else {
			return wp_send_json( __( 'Invalid Request, Incorrect Access Key', 'kadence-cloud' ) );
		}
	}
	/**
	 * Retrieves a collection of objects.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
	 */
	public function get_info( $request ) {
		if ( $this->check_access( $request ) ) {
			$settings = json_decode( get_option( 'kadence_cloud' ), true );
			if ( isset( $settings['expires'] ) && ! empty( $settings['expires'] ) ) {
				if ( 'day' === $settings['expires'] ) {
					$expires = DAY_IN_SECONDS;
				} elseif ( 'week' === $settings['expires'] ) {
					$expires = WEEK_IN_SECONDS;
				} else {
					$expires = MONTH_IN_SECONDS;
				}
			} else {
				$expires = MONTH_IN_SECONDS;
			}
			$key  = $request->get_param( self::PROP_KEY );
			$name = ( isset( $settings['cloud_name'] ) && ! empty( $settings['cloud_name'] ) ? wp_kses_post( $settings['cloud_name'] ) : wp_kses_post( get_bloginfo( 'name' ) ) );

			// Find if collection has been set as pattern hub name.
			if ( isset( $settings['access_keys'] ) ) {
				foreach ( $settings['access_keys'] as $access_key ) {
					if ( ! isset( $access_key['useAsPatternHubName'] ) ) {
						continue;
					}

					if ( ! isset( $access_key['key'] ) ) {
						continue;
					}

					if ( ! isset( $access_key['collections'] ) && empty( $access_key['collections'] ) ) {
						continue;
					}

					if ( $access_key['key'] === $key && $access_key['useAsPatternHubName'] === true ) {
						$collection = get_term_by( 'id', $access_key['collections'], 'kadence-cloud-collections' );
						if ( isset( $collection->name ) && ! empty( $collection->name ) ) {
							$name = $collection->name;
						}
					}
				}
			}

			$info = array(
				'name'    => $name,
				'slug'    => sanitize_title( md5( get_bloginfo( 'url' ) . $key ) ),
				'refresh' => ( isset( $settings['expires'] ) && ! empty( $settings['expires'] ) ? wp_kses_post( $settings['expires'] ) : 'month' ),
				'expires' => gmdate( 'Y-m-d H:i:s', strtotime( current_time( 'mysql' ) ) + $expires ),
			);
			$info = apply_filters( 'kadence_cloud_library_info_args', $info, $request );
			return wp_send_json( $info );

		} else {
			return wp_send_json( __( 'Invalid Request, Incorrect Access Key', 'kadence-cloud' ) );
		}
	}

	/**
	 * Retrieves a collection of objects.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
	 */
	public function get_items( $request ) {
		$key         = $request->get_param( self::PROP_KEY );
		$per_page    = $request->get_param( self::PROP_RETURN_COUNT );
		$page_number = $request->get_param( self::PROP_RETURN_PAGE );

		if ( $this->check_access( $request ) ) {
			$request_extras = apply_filters( 'kadence_cloud_rest_request_extras', array(), $request );
			if ( ! empty( $per_page ) ) {
				$per_page = absint( $per_page );
			} else {
				$per_page = -1;
			}
			if ( ! empty( $page_number ) ) {
				$page_number = absint( $page_number );
			} else {
				$page_number = 1;
			}
			add_filter( 'kadence_blocks_css_output_media_queries', '__return_false' );
			return $this->get_templates( $per_page, $page_number, $key, $request_extras );

		} else {
			return wp_send_json( __( 'Invalid Request, Incorrect Access Key', 'kadence-cloud' ) );
		}
	}
	/**
	 * Check if the request should get access to the files.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return Boolean true or false based on if access should be granted.
	 */
	public function check_access( $request ) {
		$access   = false;
		$key      = $request->get_param( self::PROP_KEY );
		$keys     = array();
		$settings = json_decode( get_option( 'kadence_cloud' ), true );
		if ( isset( $settings['access_keys'] ) && ! empty( $settings['access_keys'] ) && is_array( $settings['access_keys'] ) ) {
			$access_keys = array();
			if ( isset( $settings['access_keys'][0] ) && is_array( $settings['access_keys'][0] ) ) {
				foreach ( $settings['access_keys'] as $the_key => $the_keys ) {
					if ( $the_keys['key'] === $key ) {
						$access = true;
						break;
					}
				}
			} else {
				$access_keys = $settings['access_keys'];
				if ( is_array( $access_keys ) && ! empty( $access_keys ) && in_array( $key, $access_keys ) ) {
					$access = true;
				}
			}
		}
		return apply_filters( 'kadence_cloud_rest_request_access', $access, $key, $request );
	}
	/**
	 * Retrieves a collection of objects.
	 */
	public function get_template_array( $template_query, $request_extras ) {
		$library             = array();
		$fallback_image_args = apply_filters(
			'kadence_cloud_post_fallback_image_args',
			array(
				'w' => intval( 1280 ),
				'h' => intval( 800 ),
			)
		);
		if ( $template_query->have_posts() ) :
			while ( $template_query->have_posts() ) :
				$template_query->the_post();
				global $post;
				$slug_base = sanitize_title( md5( get_bloginfo( 'url' ) ) );
				$slug      = $slug_base . '-' . get_the_ID();
				if ( $slug ) {
					$library[ $slug ]         = array();
					$library[ $slug ]['slug'] = $slug;
					$library[ $slug ]['id']   = get_the_ID();
					$library[ $slug ]['name'] = the_title_attribute( 'echo=0' );
					$terms_array              = array();
					if ( get_the_terms( $post, 'kadence-cloud-categories' ) ) {
						$terms = get_the_terms( $post, 'kadence-cloud-categories' );
						if ( is_array( $terms ) ) {
							foreach ( $terms as $key => $value ) {
								$terms_array[ $value->slug ] = $value->name;
							}
						}
					}
					$library[ $slug ]['categories'] = $terms_array;
					$keywords_array                 = array();
					if ( get_the_terms( $post, 'kadence-cloud-keywords' ) ) {
						$terms = get_the_terms( $post, 'kadence-cloud-keywords' );
						if ( is_array( $terms ) ) {
							foreach ( $terms as $key => $value ) {
								$keywords_array[] = $value->name;
							}
						}
					}
					$library[ $slug ]['keywords'] = $keywords_array;
					if ( ! empty( $request_extras ) && is_array( $request_extras ) ) {
						foreach ( $request_extras as $key => $data ) {
							$library[ $slug ][ $key ] = $data;
						}
					}
					$post_extras = apply_filters( 'kadence_cloud_post_extra_args', array(), $post, $request_extras );
					if ( ! empty( $post_extras ) && is_array( $post_extras ) ) {
						foreach ( $post_extras as $key => $data ) {
							$library[ $slug ][ $key ] = $data;
						}
					}
					$library[ $slug ]['pro']    = apply_filters( 'kadence_cloud_post_is_pro', false, $post, $request_extras );
					$library[ $slug ]['locked'] = apply_filters( 'kadence_cloud_post_is_locked', false, $post, $request_extras );
					if ( apply_filters( 'kadence_cloud_post_send_content', true, $post, $request_extras ) ) {
						$library[ $slug ]['content'] = $post->post_content;
						if ( $form_ids = $this->has_form( $post->post_content ) ) {
							$library[ $slug ]['forms'] = $this->get_forms( $form_ids );
						}
					}
					if ( apply_filters( 'kadence_cloud_post_send_html_content', false, $post, $request_extras ) ) {
						global $wp_styles, $wp_scripts;
						$wp_styles  = new WP_Styles();
						$wp_scripts = new WP_Scripts();
						if ( class_exists( 'Kadence_Blocks_CSS' ) ) {
							$kadence_blocks_css = \Kadence_Blocks_CSS::get_instance();
							if ( method_exists( $kadence_blocks_css, 'frontend_block_css' ) ) {
								$kadence_blocks_css::$styles        = array();
								$kadence_blocks_css::$head_styles   = array();
								$kadence_blocks_css::$custom_styles = array();
							}
						}
						if ( class_exists( 'Kadence_Blocks_Frontend' ) ) {
							$kadence_blocks = \Kadence_Blocks_Frontend::get_instance();
							if ( method_exists( $kadence_blocks, 'frontend_build_css' ) ) {
								$kadence_blocks->frontend_build_css( $post );
							}
							if ( class_exists( 'Kadence_Blocks_Pro_Frontend' ) ) {
								$kadence_blocks_pro = \Kadence_Blocks_Pro_Frontend::get_instance();
								if ( method_exists( $kadence_blocks_pro, 'frontend_build_css' ) ) {
									$kadence_blocks_pro->frontend_build_css( $post );
								}
							}
						}
						if ( class_exists( 'Kadence_Blocks_CSS' ) ) {
							$kadence_blocks_css = \Kadence_Blocks_CSS::get_instance();
							if ( method_exists( $kadence_blocks_css, 'frontend_block_css' ) ) {
								$kadence_blocks_css->frontend_block_css();
							}
						}
						ob_start();
						wp_print_styles();
						echo do_blocks( $post->post_content );
						$wp_scripts->print_scripts();
						$library[ $slug ]['html'] = ob_get_clean();
					}
					$library[ $slug ]['description'] = $post->post_excerpt;
					$image_meta                      = get_post_meta( get_the_ID(), '_kc_preview_image', true );
					if ( has_post_thumbnail() ) {
						$image                      = wp_get_attachment_image_src( get_post_thumbnail_id( $post ), 'full' );
						$library[ $slug ]['image']  = $image[0];
						$library[ $slug ]['imageW'] = $image[1];
						$library[ $slug ]['imageH'] = $image[2];
					} else if ( ! empty( $image_meta['url'] ) ) {
						$library[ $slug ]['image']  = $image_meta['url'];
						$library[ $slug ]['imageW'] = $image_meta['width'];
						$library[ $slug ]['imageH'] = $image_meta['height'];
					} else {
						$library[ $slug ]['image']  = add_query_arg( $fallback_image_args, 'https://s0.wordpress.com/mshots/v1/' . rawurlencode( esc_url( get_permalink() ) ) );
						$library[ $slug ]['imageW'] = $fallback_image_args['w'];
						$library[ $slug ]['imageH'] = $fallback_image_args['h'];
					}
					$library[ $slug ] = apply_filters( 'kadence_cloud_individual_library_item_data', $library[ $slug ], $post, $request_extras );
				}
			endwhile;
		endif;
		return $library;
	}
	/**
	 * Get Forms from ids.
	 *
	 * @param array $form_ids array of form ids.
	 */
	public function get_forms( $form_ids ) {
		$forms = array();
		foreach ( $form_ids as $form_id ) {
			$form = get_post( $form_id );
			if ( ! empty( $form ) && 'kadence_form' === $form->post_type ) {
				$forms[ $form_id ] = array(
					'id'      => $form_id,
					'name'    => $form->post_title,
					'content' => $form->post_content,
					'meta'    => array(),
				);
				$meta = get_post_meta( $form_id );
				foreach ( $meta as $key => $value ) {
					if ( 0 === strpos( $key, '_kad_form_' ) ) {
						$forms[ $form_id ]['meta'][ $key ] = maybe_unserialize( $value[0] );
					}
				}
			}
		}
		return $forms;
	}
	/**
	 * Check if pattern has Advanced form
	 */
	public function has_form( $content ) {
		$blocks = parse_blocks( $content );
		if ( ! is_array( $blocks ) || empty( $blocks ) ) {
			return array();
		}
		$form_ids = array();
		foreach ( $blocks as $indexkey => $block ) {
			if ( ! is_object( $block ) && is_array( $block ) && isset( $block['blockName'] ) ) {
				if ( 'kadence/advanced-form' === $block['blockName'] ) {
					if ( isset( $block['attrs'] ) && is_array( $block['attrs'] ) ) {
						$blockattr = $block['attrs'];
						if ( isset( $blockattr['id'] ) ) {
							$form_ids[] = $blockattr['id'];
						}
					}
				}
				if ( ! empty( $block['innerBlocks'] ) && is_array( $block['innerBlocks'] ) ) {
					$forms_to_add = $this->blocks_cycle_through( $block['innerBlocks'] );
					if ( ! empty( $forms_to_add ) ) {
						$form_ids = array_merge( $form_ids, $forms_to_add );
					}
				}
			}
		}
		return $form_ids;
	}
	/**
	 * Builds css for inner blocks
	 *
	 * @param array $inner_blocks array of inner blocks.
	 */
	public function blocks_cycle_through( $inner_blocks ) {
		$forms_to_add = array();
		foreach ( $inner_blocks as $in_indexkey => $inner_block ) {
			if ( 'kadence/advanced-form' === $inner_block['blockName'] ) {
				if ( isset( $inner_block['attrs'] ) && is_array( $inner_block['attrs'] ) ) {
					$blockattr = $inner_block['attrs'];
					if ( isset( $blockattr['id'] ) ) {
						$forms_to_add[] = $blockattr['id'];
					}
				}
			}
			if ( ! empty( $inner_block['innerBlocks'] ) && is_array( $inner_block['innerBlocks'] ) ) {
				$other_forms_to_add = $this->blocks_cycle_through( $inner_block['innerBlocks'] );
				if ( ! empty( $other_forms_to_add ) ) {
					$forms_to_add = array_merge( $forms_to_add, $other_forms_to_add );
				}
			}
		}
		return $forms_to_add;
	}
	/**
	 * Retrieves a collection of objects.
	 */
	public function get_templates( $post_per_page = -1, $page_number = 1, $key = '', $request_extras = array() ) {
		$args     = array(
			'post_type'      => 'kadence_cloud',
			'post_status'    => 'publish',
			'posts_per_page' => $post_per_page,
			'order'          => 'ASC',
			'meta_key'       => apply_filters( 'kadence_cloud_order_by_popular', false ) ? '_kc_import_count' : '',
			'orderby'        => apply_filters( 'kadence_cloud_order_by_popular', false ) ? 'meta_value_num' : 'menu_order',
			'offset'         => ( 1 < $page_number && -1 !== $post_per_page ? ( $page_number * $post_per_page ) : 0 ),
		);
		$settings = json_decode( get_option( 'kadence_cloud' ), true );
		if ( isset( $settings['access_keys'] ) && ! empty( $settings['access_keys'] ) && is_array( $settings['access_keys'] ) ) {
			if ( isset( $settings['access_keys'][0] ) && is_array( $settings['access_keys'][0] ) ) {
				foreach ( $settings['access_keys'] as $the_key => $the_keys ) {
					if ( $the_keys['key'] === $key ) {
						if ( isset( $the_keys['collections'] ) && ! empty( $the_keys['collections'] ) ) {
							$args['tax_query'] = array(
								array(
									'taxonomy' => 'kadence-cloud-collections',
									'field'    => 'id',
									'terms'    => explode( ',', $the_keys['collections'] ),
								),
							);
						}
						break;
					}
				}
			}
		}
		$args      = apply_filters( 'kadence_cloud_template_query_args', $args, $key, $request_extras );
		$templates = new WP_Query( $args );
		$library   = $this->get_template_array( $templates, $request_extras );

		wp_send_json( $library );
	}
	/**
	 * Extracts a string between two strings.
	 * 
	 * @param string $string The string to search.
	 * @param string $start The string to start with.
	 * @param string $end The string to end with.
	 * @param string $verify The string to verify.
	 */
	public function get_string_inbetween( $string, $start, $end, $verify ) {
		if ( strpos( $string, $verify ) == 0 ) {
			return '';
		}
		$ini = strpos( $string, $start );
		if ( $ini == 0 ) {
			return '';
		}
		$ini += strlen( $start );
		$len = strpos( $string, $end, $ini ) - $ini;
		return substr( $string, $ini, $len );
	}
	/**
	 * Extracts a string between two strings when the verify is within.
	 * 
	 * @param string $string The string to search.
	 * @param string $start The string to start with.
	 * @param string $end The string to end with.
	 * @param string $verify The string to verify.
	 * @param string $from The string to start from.
	 */
	public function get_string_inbetween_when( $string, $start, $end, $verify, $from ) {
		if ( strpos( $string, $verify ) == 0 ) {
			return '';
		}
		$ini = strpos( $string, $start, $from );
		if ( $ini == 0 ) {
			return '';
		}
		$ini += strlen( $start );
		$len        = strpos( $string, $end, $ini ) - $ini;
		$sub_string = substr( $string, $ini, $len );
		if ( strpos( $sub_string, $verify ) == 0 ) {
			return $this->get_string_inbetween_when( $string, $start, $end, $verify, $ini );
		}
		return $sub_string;
	}
	/**
	 * Retrieves the query params for the search results collection.
	 *
	 * @return array Collection parameters.
	 */
	public function get_collection_params() {
		$query_params = parent::get_collection_params();

		$query_params[ self::PROP_KEY ] = array(
			'description' => __( 'The request key.', 'kadence-cloud' ),
			'type'        => 'string',
		);

		$query_params[ self::PROP_SITE ] = array(
			'description' => __( 'The request website.', 'kadence-cloud' ),
			'type'        => 'string',
		);

		$query_params[ self::PROP_RETURN_COUNT ] = array(
			'description' => __( 'Items to return.', 'kadence-cloud' ),
			'type'        => 'string',
		);

		$query_params[ self::PROP_RETURN_PAGE ] = array(
			'description' => __( 'The Page to return.', 'kadence-cloud' ),
			'type'        => 'string',
		);

		return $query_params;
	}
	/**
	 * Retrieves the query params for the search results collection.
	 *
	 * @return array Collection parameters.
	 */
	public function get_analytics_params() {
		$query_params = parent::get_collection_params();

		$query_params[ self::PROP_KEY ] = array(
			'description' => __( 'The request key.', 'kadence-cloud' ),
			'type'        => 'string',
		);
		$query_params[ self::PROP_ID ]  = array(
			'description' => __( 'The item id.', 'kadence-cloud' ),
			'type'        => 'string',
		);

		$query_params[ self::PROP_TYPE ] = array(
			'description' => __( 'The Item Type.', 'kadence-cloud' ),
			'type'        => 'string',
		);

		$query_params[ self::PROP_STYLE ] = array(
			'description' => __( 'The item style.', 'kadence-cloud' ),
			'type'        => 'string',
		);

		return $query_params;
	}
}
