<?php
/**
 * Plugin Name: Sity Cloud SaaS Billing
 * Description: SaaS-style billing portal (plans, upgrade/downgrade, trials, webhooks) using WooCommerce (+ Paid Memberships Pro إذا موجود).
 * Author: Sity Cloud
 * Version: 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

if (!class_exists('SityCloud_SaaS_Billing')) {

    class SityCloud_SaaS_Billing {

        public function __construct() {
            // Shortcode للعميل
            add_shortcode('sity_billing_portal', [$this, 'render_billing_portal']);

            // صفحة الأدمن
            add_action('admin_menu', [$this, 'register_admin_menu']);

            // AJAX لتغيير الباقة
            add_action('wp_ajax_sity_change_plan', [$this, 'handle_change_plan']);
            add_action('wp_ajax_nopriv_sity_change_plan', [$this, 'handle_change_plan_guest']);

            // REST API Webhook
            add_action('rest_api_init', [$this, 'register_rest_routes']);

            // حقول إضافية في منتج WooCommerce لتحديد الباقات
            add_action('woocommerce_product_options_general_product_data', [$this, 'add_product_plan_fields']);
            add_action('woocommerce_process_product_meta', [$this, 'save_product_plan_fields']);
        }

        /* ===========================
         *  Helpers
         * =========================== */

        // جيب كل المنتجات اللي متعلّمة كباقات Billing
        public function get_billing_products() {
            if (!class_exists('WC_Product')) {
                return [];
            }

            $args = [
                'status' => ['publish'],
                'limit'  => -1,
                'type'   => ['simple', 'subscription', 'variable-subscription', 'variable', 'membership'],
                'meta_key'   => '_sity_is_billing_plan',
                'meta_value' => 'yes',
            ];

            $products = wc_get_products($args);

            // Sort حسب السعر
            usort($products, function($a, $b) {
                return floatval($a->get_price()) <=> floatval($b->get_price());
            });

            return $products;
        }

        // جيب الباقة الحالية للمستخدم من آخر طلب/عضوية
        public function get_user_current_plan($user_id) {
            if (!class_exists('WooCommerce')) {
                return null;
            }

            $plan = [
                'product_id' => 0,
                'product_name' => '',
                'price' => 0,
            ];

            // حاول من Paid Memberships Pro إن وجد
            if (function_exists('pmpro_getMembershipLevelForUser')) {
                $level = pmpro_getMembershipLevelForUser($user_id);
                if ($level && isset($level->name)) {
                    $plan['product_name'] = $level->name . ' (PMPro)';
                }
            }

            // آخر أوردر من WooCommerce
            $customer_orders = wc_get_orders([
                'customer_id' => $user_id,
                'limit'       => 1,
                'orderby'     => 'date',
                'order'       => 'DESC',
                'status'      => ['completed', 'processing', 'on-hold'],
            ]);

            if (!empty($customer_orders)) {
                $order = $customer_orders[0];
                foreach ($order->get_items() as $item) {
                    $product = $item->get_product();
                    if (!$product) continue;

                    // لازم يكون متعلّم إنه Billing plan
                    $is_plan = get_post_meta($product->get_id(), '_sity_is_billing_plan', true);
                    if ($is_plan === 'yes') {
                        $plan['product_id'] = $product->get_id();
                        $plan['product_name'] = $product->get_name();
                        $plan['price'] = floatval($product->get_price());
                        break;
                    }
                }
            }

            return $plan['product_id'] ? $plan : null;
        }

        // Text label حسب سعر المنتج مقارنة بالباقة الحالية
        public function get_plan_relation_label($price, $current_price) {
            if ($current_price === null) {
                return 'new';
            }
            if ($price > $current_price) {
                return 'upgrade';
            }
            if ($price < $current_price) {
                return 'downgrade';
            }
            return 'current';
        }

        /* ===========================
         *  Shortcode: [sity_billing_portal]
         * =========================== */

        public function render_billing_portal($atts) {
            if (!is_user_logged_in()) {
                return '<div class="sity-billing-portal sity-billing-portal-guest">
                    <p>من فضلك قم بتسجيل الدخول لعرض صفحة الفوترة الخاصة بك.</p>
                </div>';
            }

            if (!class_exists('WooCommerce')) {
                return '<div class="sity-billing-portal sity-billing-portal-error">
                    <p>هذه الصفحة تحتاج إلى تنصيب وتفعيل <strong>WooCommerce</strong> أولاً.</p>
                </div>';
            }

            $user_id     = get_current_user_id();
            $current_plan = $this->get_user_current_plan($user_id);
            $products     = $this->get_billing_products();
            $nonce        = wp_create_nonce('sity_change_plan_nonce');
            $checkout_url = wc_get_checkout_url();

            ob_start();
            ?>

            <style>
                .sity-billing-portal {
                    direction: rtl;
                    font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
                    max-width: 1100px;
                    margin: 40px auto;
                    padding: 24px;
                    border-radius: 24px;
                    background: radial-gradient(circle at top left, rgba(139, 92, 246, 0.35), transparent 55%),
                                linear-gradient(145deg, rgba(17, 24, 39, 0.98), rgba(17, 24, 39, 0.9));
                    color: #f9fafb;
                    box-shadow: 0 24px 60px rgba(0, 0, 0, 0.45);
                    position: relative;
                    overflow: hidden;
                }

                .sity-billing-portal::before {
                    content: "";
                    position: absolute;
                    inset: 0;
                    background: radial-gradient(circle at bottom right, rgba(79, 70, 229, 0.25), transparent 60%);
                    opacity: 0.9;
                    pointer-events: none;
                }

                .sity-billing-inner {
                    position: relative;
                    z-index: 1;
                }

                .sity-billing-header {
                    display: flex;
                    flex-wrap: wrap;
                    justify-content: space-between;
                    gap: 16px;
                    align-items: flex-start;
                    margin-bottom: 24px;
                }

                .sity-billing-title {
                    font-size: 1.7rem;
                    font-weight: 700;
                    display: flex;
                    align-items: center;
                    gap: 10px;
                }

                .sity-billing-title span.badge {
                    font-size: 0.78rem;
                    padding: 4px 10px;
                    border-radius: 999px;
                    border: 1px solid rgba(251, 191, 36, 0.5);
                    background: rgba(31, 41, 55, 0.8);
                    color: #facc15;
                }

                .sity-billing-subtitle {
                    font-size: 0.92rem;
                    color: #d1d5db;
                    max-width: 420px;
                }

                .sity-current-plan {
                    margin-bottom: 24px;
                    padding: 16px 18px;
                    border-radius: 16px;
                    background: linear-gradient(135deg, rgba(15, 23, 42, 0.95), rgba(30, 64, 175, 0.9));
                    border: 1px solid rgba(129, 140, 248, 0.6);
                    display: flex;
                    flex-wrap: wrap;
                    gap: 10px 20px;
                    align-items: center;
                    justify-content: space-between;
                }

                .sity-current-plan-main {
                    display: flex;
                    flex-wrap: wrap;
                    gap: 10px 20px;
                    align-items: center;
                }

                .sity-current-plan-label {
                    font-size: 0.9rem;
                    color: #cbd5f5;
                }

                .sity-current-plan-name {
                    font-weight: 600;
                    font-size: 1rem;
                    color: #f9fafb;
                }

                .sity-current-plan-price {
                    font-size: 0.9rem;
                    color: #e5e7eb;
                }

                .sity-badge-trial {
                    padding: 4px 10px;
                    border-radius: 999px;
                    font-size: 0.78rem;
                    border: 1px dashed rgba(96, 165, 250, 0.8);
                    background: rgba(15, 23, 42, 0.85);
                    color: #bfdbfe;
                }

                .sity-billing-grid {
                    display: grid;
                    grid-template-columns: repeat(3, minmax(0, 1fr));
                    gap: 16px;
                }

                @media (max-width: 960px) {
                    .sity-billing-grid {
                        grid-template-columns: repeat(2, minmax(0, 1fr));
                    }
                }
                @media (max-width: 720px) {
                    .sity-billing-header {
                        flex-direction: column;
                    }
                    .sity-billing-grid {
                        grid-template-columns: 1fr;
                    }
                }

                .sity-plan-card {
                    position: relative;
                    padding: 18px 16px;
                    border-radius: 18px;
                    background: rgba(15, 23, 42, 0.9);
                    border: 1px solid rgba(55, 65, 81, 0.9);
                    backdrop-filter: blur(18px);
                    display: flex;
                    flex-direction: column;
                    gap: 10px;
                    min-height: 210px;
                    overflow: hidden;
                }

                .sity-plan-card.upgrade {
                    border-color: rgba(96, 165, 250, 0.9);
                    box-shadow: 0 0 25px rgba(59, 130, 246, 0.45);
                }

                .sity-plan-card.downgrade {
                    border-color: rgba(75, 85, 99, 0.9);
                    opacity: 0.9;
                }

                .sity-plan-card.current {
                    border-color: rgba(129, 140, 248, 0.9);
                    box-shadow: 0 0 24px rgba(129, 140, 248, 0.4);
                }

                .sity-plan-label {
                    font-size: 0.8rem;
                    padding: 3px 10px;
                    border-radius: 999px;
                    display: inline-flex;
                    align-items: center;
                    gap: 6px;
                }

                .sity-plan-label.upgrade {
                    background: rgba(22, 163, 74, 0.12);
                    color: #bbf7d0;
                    border: 1px solid rgba(34, 197, 94, 0.45);
                }

                .sity-plan-label.downgrade {
                    background: rgba(30, 64, 175, 0.12);
                    color: #bfdbfe;
                    border: 1px solid rgba(59, 130, 246, 0.4);
                }

                .sity-plan-label.current {
                    background: rgba(129, 140, 248, 0.18);
                    color: #e0e7ff;
                    border: 1px solid rgba(129, 140, 248, 0.7);
                }

                .sity-plan-name {
                    font-size: 1.05rem;
                    font-weight: 600;
                }

                .sity-plan-price-row {
                    display: flex;
                    align-items: baseline;
                    gap: 6px;
                }

                .sity-plan-price-main {
                    font-size: 1.3rem;
                    font-weight: 700;
                    color: #f9fafb;
                }

                .sity-plan-price-suffix {
                    font-size: 0.8rem;
                    color: #9ca3af;
                }

                .sity-plan-meta {
                    font-size: 0.8rem;
                    color: #9ca3af;
                }

                .sity-plan-badges {
                    display: flex;
                    flex-wrap: wrap;
                    gap: 6px;
                    font-size: 0.75rem;
                }

                .sity-pill {
                    padding: 3px 8px;
                    border-radius: 999px;
                    border: 1px solid rgba(55, 65, 81, 0.9);
                    background: rgba(15, 23, 42, 0.9);
                    color: #e5e7eb;
                }

                .sity-pill.trial {
                    border-color: rgba(56, 189, 248, 0.8);
                    color: #a5f3fc;
                }

                .sity-pill.auto-renew {
                    border-color: rgba(129, 140, 248, 0.9);
                    color: #e0e7ff;
                }

                .sity-plan-footer {
                    margin-top: auto;
                    display: flex;
                    justify-content: space-between;
                    align-items: center;
                    gap: 10px;
                }

                .sity-plan-btn {
                    appearance: none;
                    border: none;
                    border-radius: 999px;
                    padding: 8px 14px;
                    font-size: 0.85rem;
                    font-weight: 600;
                    cursor: pointer;
                    display: inline-flex;
                    align-items: center;
                    gap: 6px;
                    transition: transform 0.1s ease, box-shadow 0.1s ease, background 0.15s ease;
                    white-space: nowrap;
                }

                .sity-plan-btn.primary {
                    background: linear-gradient(135deg, #8b5cf6, #6366f1);
                    color: #f9fafb;
                    box-shadow: 0 10px 25px rgba(129, 140, 248, 0.5);
                }

                .sity-plan-btn.secondary {
                    background: rgba(15, 23, 42, 0.95);
                    color: #e5e7eb;
                    border: 1px solid rgba(55, 65, 81, 0.9);
                }

                .sity-plan-btn:disabled {
                    opacity: 0.6;
                    cursor: default;
                    box-shadow: none;
                    transform: none;
                }

                .sity-plan-btn:not(:disabled):hover {
                    transform: translateY(-1px);
                    box-shadow: 0 12px 30px rgba(129, 140, 248, 0.6);
                }

                .sity-plan-btn:not(:disabled):active {
                    transform: translateY(1px);
                    box-shadow: 0 5px 15px rgba(15, 23, 42, 0.8);
                }

                .sity-billing-meta-row {
                    display: flex;
                    flex-wrap: wrap;
                    gap: 8px;
                    font-size: 0.75rem;
                    color: #9ca3af;
                    margin-top: 14px;
                }

                .sity-dot {
                    width: 6px;
                    height: 6px;
                    border-radius: 999px;
                    background: #22c55e;
                }

                .sity-dot.red {
                    background: #ef4444;
                }

                .sity-webhook-status {
                    display: inline-flex;
                    align-items: center;
                    gap: 6px;
                    font-size: 0.78rem;
                }

                .sity-billing-loading {
                    font-size: 0.85rem;
                    color: #e5e7eb;
                }

            </style>

            <div class="sity-billing-portal">
                <div class="sity-billing-inner">
                    <div class="sity-billing-header">
                        <div>
                            <div class="sity-billing-title">
                                <span>إدارة الباقات والفوترة</span>
                                <span class="badge">SaaS Billing</span>
                            </div>
                            <div class="sity-billing-subtitle">
                                راقب باقتك الحالية، جرّب الباقات الأعلى، وانقل بين الباقات بدون صداع.
                           
                            </div>
                        </div>
                        <div>
                            <div class="sity-webhook-status">
                                <span class="sity-dot"></span>
                                <span>Webhook API جاهز لاستقبال التحديثات</span>
                            </div>
                        </div>
                    </div>

                    <div class="sity-current-plan">
                        <div class="sity-current-plan-main">
                            <div>
                                <div class="sity-current-plan-label">الباقة الحالية</div>
                                <div class="sity-current-plan-name">
                                    <?php
                                    if ($current_plan) {
                                        echo esc_html($current_plan['product_name']);
                                    } else {
                                        echo 'لا توجد باقة مفعّلة حاليًا';
                                    }
                                    ?>
                                </div>
                            </div>
                            <?php if ($current_plan && $current_plan['price'] !== null): ?>
                                <div class="sity-current-plan-price">
                                    <?php echo esc_html(number_format_i18n($current_plan['price'], 2)); ?> جـ / شهر
                                </div>
                            <?php endif; ?>
                        </div>

                        <div>
                            <?php if (function_exists('pmpro_getMembershipLevelForUser')): ?>
                                <?php $level = pmpro_getMembershipLevelForUser($user_id); ?>
                                <?php if ($level): ?>
                                    <span class="sity-badge-trial">
                                        مستوى العضوية: <?php echo esc_html($level->name); ?>
                                    </span>
                                <?php else: ?>
                                    
                                <?php endif; ?>
                            <?php else: ?>
                                <span class="sity-badge-trial">
                                   التكامل لم يتم اكتشافه
                                </span>
                            <?php endif; ?>
                        </div>
                    </div>

                    <?php if (empty($products)): ?>
                        <p>لا توجد باقات معرفة حتى الآن. فعّل خيار "هذه باقة Billing" من إعدادات المنتج في .</p>
                    <?php else: ?>
                        <div class="sity-billing-grid">
                            <?php
                            $current_price = $current_plan ? $current_plan['price'] : null;

                            foreach ($products as $product):
                                $pid          = $product->get_id();
                                $price        = floatval($product->get_price());
                                $relation     = $this->get_plan_relation_label($price, $current_price);
                                $trial_days   = get_post_meta($pid, '_sity_trial_days', true);
                                $auto_renew   = get_post_meta($pid, '_sity_auto_renew', true) === 'yes';
                                $tagline      = get_post_meta($pid, '_sity_plan_tagline', true);

                                $card_class = 'sity-plan-card ' . $relation;
                                ?>
                                <div class="<?php echo esc_attr($card_class); ?>">
                                    <div class="sity-plan-label <?php echo esc_attr($relation); ?>">
                                        <?php
                                        if ($relation === 'upgrade') {
                                            echo 'ترقية الباقة';
                                        } elseif ($relation === 'downgrade') {
                                            echo 'تخفيض الباقة';
                                        } elseif ($relation === 'current') {
                                            echo 'الباقة الحالية';
                                        } else {
                                            echo 'باقة متاحة';
                                        }
                                        ?>
                                    </div>

                                    <div class="sity-plan-name">
                                        <?php echo esc_html($product->get_name()); ?>
                                    </div>

                                    <div class="sity-plan-price-row">
                                        <span class="sity-plan-price-main">
                                            <?php echo esc_html(number_format_i18n($price, 2)); ?>
                                        </span>
                                        <span class="sity-plan-price-suffix">جـ / شهر</span>
                                    </div>

                                    <?php if (!empty($tagline)): ?>
                                        <div class="sity-plan-meta">
                                            <?php echo esc_html($tagline); ?>
                                        </div>
                                    <?php endif; ?>

                                    <div class="sity-plan-badges">
                                        <?php if (!empty($trial_days)): ?>
                                            <span class="sity-pill trial">
                                                تجربة مجانية <?php echo intval($trial_days); ?> يوم
                                            </span>
                                        <?php endif; ?>

                                        <?php if ($auto_renew): ?>
                                            <span class="sity-pill auto-renew">
                                                تجديد تلقائي
                                            </span>
                                        <?php endif; ?>

                                        <span class="sity-pill">
                                            ID: <?php echo esc_html($pid); ?>
                                        </span>
                                    </div>

                                    <div class="sity-plan-footer">
                                        <button
                                            class="sity-plan-btn <?php echo $relation === 'current' ? 'secondary' : 'primary'; ?>"
                                            data-plan-id="<?php echo esc_attr($pid); ?>"
                                            data-relation="<?php echo esc_attr($relation); ?>"
                                            <?php echo $relation === 'current' ? 'disabled' : ''; ?>
                                        >
                                            <?php
                                            if ($relation === 'upgrade') {
                                                echo 'ترقية لهذه الباقة';
                                            } elseif ($relation === 'downgrade') {
                                                echo 'التحويل لهذه الباقة';
                                            } elseif ($relation === 'current') {
                                                echo 'هذه هي باقتك الحالية';
                                            } else {
                                                echo 'اشترك في هذه الباقة';
                                            }
                                            ?>
                                            <span>→</span>
                                        </button>

                                        <a href="https://sity.cloud/pricing/" style="font-size:0.78rem;color:#9ca3af;text-decoration:underline;">
                                            تفاصيل الباقة
                                        </a>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>

                    <div class="sity-billing-meta-row">
                        <div>• يتم إنشاء الطلبات وعمليات الدفع   .</div>
                        <div>• يمكنك استخدام Webhooks لربط الباقات مع أنظمة خارجية (CRM، أنظمة داخلية، إلخ).</div>
                    </div>

                    <div id="sity-billing-message" class="sity-billing-loading" style="display:none;margin-top:10px;"></div>

                </div>
            </div>

            <script>
                (function() {
                    const buttons = document.querySelectorAll('.sity-plan-btn[data-plan-id]');
                    const msgEl = document.getElementById('sity-billing-message');

                    function showMessage(text) {
                        if (!msgEl) return;
                        msgEl.style.display = 'block';
                        msgEl.textContent = text;
                    }

                    buttons.forEach(function(btn) {
                        btn.addEventListener('click', function(e) {
                            e.preventDefault();
                            const planId = this.getAttribute('data-plan-id');
                            const relation = this.getAttribute('data-relation');
                            if (!planId) return;

                            this.disabled = true;
                            showMessage('جاري تجهيز بوابة الدفع للباقة المختارة...');

                            const formData = new FormData();
                            formData.append('action', 'sity_change_plan');
                            formData.append('plan_id', planId);
                            formData.append('nonce', '<?php echo esc_js($nonce); ?>');

                            fetch('<?php echo esc_url(admin_url('admin-ajax.php')); ?>', {
                                method: 'POST',
                                credentials: 'same-origin',
                                body: formData
                            })
                            .then(res => res.json())
                            .then(data => {
                                if (data && data.success && data.data && data.data.redirect) {
                                    window.location.href = data.data.redirect;
                                } else {
                                    showMessage(data && data.data && data.data.message ? data.data.message : 'حدث خطأ غير متوقع.');
                                    btn.disabled = false;
                                }
                            })
                            .catch(err => {
                                console.error(err);
                                showMessage('تعذر الاتصال بالخادم. حاول مرة أخرى.');
                                btn.disabled = false;
                            });
                        });
                    });
                })();
            </script>

            <?php
            return ob_get_clean();
        }

        /* ===========================
         *  AJAX: change plan
         * =========================== */

        public function handle_change_plan() {
            if (!is_user_logged_in()) {
                wp_send_json_error(['message' => 'يجب تسجيل الدخول لتغيير الباقة.']);
            }

            if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'sity_change_plan_nonce')) {
                wp_send_json_error(['message' => 'طلب غير مصرح.']);
            }

            if (!class_exists('WooCommerce')) {
                wp_send_json_error(['message' => 'WooCommerce غير متوفر.']);
            }

            $plan_id = isset($_POST['plan_id']) ? intval($_POST['plan_id']) : 0;
            if (!$plan_id) {
                wp_send_json_error(['message' => 'معرّف الباقة غير صالح.']);
            }

            $product = wc_get_product($plan_id);
            if (!$product) {
                wp_send_json_error(['message' => 'لم يتم العثور على هذه الباقة.']);
            }

            $is_plan = get_post_meta($plan_id, '_sity_is_billing_plan', true);
            if ($is_plan !== 'yes') {
                wp_send_json_error(['message' => 'هذا المنتج غير معرّف كباقة Billing.']);
            }

            // تفريغ السلة ثم إضافة الباقة الجديدة
            WC()->cart->empty_cart();
            WC()->cart->add_to_cart($plan_id, 1);

            $checkout_url = wc_get_checkout_url();
            wp_send_json_success(['redirect' => $checkout_url]);
        }

        public function handle_change_plan_guest() {
            wp_send_json_error(['message' => 'قم بتسجيل الدخول أولاً ثم أعد المحاولة.']);
        }

        /* ===========================
         *  Admin Page
         * =========================== */

        public function register_admin_menu() {
            add_menu_page(
                'Sity Billing',
                'Sity Billing',
                'manage_options',
                'sity-billing-admin',
                [$this, 'render_admin_page'],
                'dashicons-money',
                56
            );
        }

        public function render_admin_page() {
            if (!current_user_can('manage_options')) {
                return;
            }

            $users = get_users([
                'role__in' => ['customer', 'subscriber'],
                'number'   => 100,
            ]);
            ?>
            <div class="wrap">
                <h1>Sity Cloud Billing – لوحة تحكم العملاء</h1>
                <p>هذه الصفحة تعرض لمحة سريعة عن العملاء والباقات الحالية الخاصة بهم.</p>

                <table class="widefat fixed striped">
                    <thead>
                    <tr>
                        <th>المستخدم</th>
                        <th>الإيميل</th>
                        <th>الباقة الحالية</th>
                        <th>السعر (ج/شهر)</th>
                        <th>عضوية </th>
                        <th>آخر طلب </th>
                    </tr>
                    </thead>
                    <tbody>
                    <?php if (empty($users)): ?>
                        <tr><td colspan="6">لا يوجد عملاء حتى الآن.</td></tr>
                    <?php else: ?>
                        <?php foreach ($users as $user):
                            $plan = $this->get_user_current_plan($user->ID);
                            $pmpro_name = '-';
                            if (function_exists('pmpro_getMembershipLevelForUser')) {
                                $level = pmpro_getMembershipLevelForUser($user->ID);
                                if ($level) {
                                    $pmpro_name = esc_html($level->name);
                                }
                            }

                            $last_order_info = '-';
                            if (class_exists('WooCommerce')) {
                                $orders = wc_get_orders([
                                    'customer_id' => $user->ID,
                                    'limit'       => 1,
                                    'orderby'     => 'date',
                                    'order'       => 'DESC'
                                ]);
                                if (!empty($orders)) {
                                    $order = $orders[0];
                                    $last_order_info = sprintf(
                                        '#%d - %s (%s)',
                                        $order->get_id(),
                                        wc_price($order->get_total()),
                                        wc_get_order_status_name($order->get_status())
                                    );
                                }
                            }
                            ?>
                            <tr>
                                <td><?php echo esc_html($user->display_name); ?></td>
                                <td><?php echo esc_html($user->user_email); ?></td>
                                <td><?php echo $plan ? esc_html($plan['product_name']) : 'لا توجد باقة'; ?></td>
                                <td><?php echo $plan ? esc_html(number_format_i18n($plan['price'], 2)) : '-'; ?></td>
                                <td><?php echo $pmpro_name; ?></td>
                                <td><?php echo $last_order_info; ?></td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                    </tbody>
                </table>

                <h2 style="margin-top:30px;">Webhook Endpoint</h2>
                <p>
                    رابط Webhook للاستخدام الخارجي (مثلاً الربط مع Stripe / خدمة أخرى):<br>
                    <code><?php echo esc_html(rest_url('sity-billing/v1/webhook')); ?></code>
                </p>
                <p>أي طلب POST يتم إرساله إلى هذا الرابط يتم تسجيله في خيار  (يمكنك قراءته برمجياً) أو التعامل معه كما تشاء بتطوير إضافي.</p>
            </div>
            <?php
        }

        /* ===========================
         *  Product Fields (WooCommerce)
         * =========================== */

        public function add_product_plan_fields() {
            echo '<div class="options_group">';
            // هل هذا المنتج باقة Billing؟
            woocommerce_wp_checkbox([
                'id'          => '_sity_is_billing_plan',
                'label'       => 'Sity Billing Plan',
                'description' => 'حدد هذا الخيار إذا كان هذا المنتج يمثل باقة Billing (SaaS).',
            ]);

            // tagline
            woocommerce_wp_text_input([
                'id'          => '_sity_plan_tagline',
                'label'       => 'وصف قصير للباقة',
                'description' => 'سطر تسويقي قصير يظهر في كارت الباقة (اختياري).',
                'desc_tip'    => true,
            ]);

            // trial days
            woocommerce_wp_text_input([
                'id'          => '_sity_trial_days',
                'label'       => 'عدد أيام التجربة المجانية',
                'description' => 'مثال: 7 – لعرض "تجربة مجانية 7 أيام". الاستخدام الكامل يتطلب دعم من إضافة الاشتراكات.',
                'type'        => 'number',
                'custom_attributes' => [
                    'min' => '0',
                    'step' => '1',
                ],
                'desc_tip'    => true,
            ]);

            // auto-renew
            woocommerce_wp_checkbox([
                'id'          => '_sity_auto_renew',
                'label'       => 'تجديد تلقائي',
                'description' => 'عرض هذه الباقة كاشتراك يتم تجديده تلقائيًا (يحتاج إعدادات من نظام الدفع).',
            ]);

            echo '</div>';
        }

        public function save_product_plan_fields($post_id) {
            $is_plan = isset($_POST['_sity_is_billing_plan']) ? 'yes' : 'no';
            update_post_meta($post_id, '_sity_is_billing_plan', $is_plan);

            if (isset($_POST['_sity_plan_tagline'])) {
                update_post_meta($post_id, '_sity_plan_tagline', sanitize_text_field($_POST['_sity_plan_tagline']));
            }

            if (isset($_POST['_sity_trial_days'])) {
                update_post_meta($post_id, '_sity_trial_days', intval($_POST['_sity_trial_days']));
            }

            $auto_renew = isset($_POST['_sity_auto_renew']) ? 'yes' : 'no';
            update_post_meta($post_id, '_sity_auto_renew', $auto_renew);
        }

        /* ===========================
         *  REST API: Webhook
         * =========================== */

        public function register_rest_routes() {
            register_rest_route('sity-billing/v1', '/webhook', [
                'methods'  => 'POST',
                'callback' => [$this, 'handle_webhook'],
                'permission_callback' => '__return_true',
            ]);
        }

        public function handle_webhook($request) {
            $payload = $request->get_params();

            // تسجيل آخر payload كـ option بسيطة – يمكنك تطويرها لتخزين في جدول مخصص
            update_option('sity_billing_last_webhook', [
                'time'    => current_time('mysql'),
                'payload' => $payload,
            ]);

            return new WP_REST_Response([
                'status' => 'ok',
                'message' => 'Webhook received',
            ], 200);
        }

    }

    new SityCloud_SaaS_Billing();
}
