<?php
/**
 * Plugin Name: Sity Cloud Profile Bridge
 * Description: لوحة بروفايل لعملاء Sity Cloud تربط بين بيانات fawry + SityCloud Wallet مع شورت كود [sitycloud_profile] + صفحة أدمن لكل العملاء.
 * Author: Sity Cloud
 * Version: 1.0.0
 * Text Domain: sitycloud-profile-bridge
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class SityCloud_Profile_Bridge {

    private static $instance = null;
    private static $css_printed = false;

    public static function get_instance() {
        if ( self::$instance === null ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        add_shortcode( 'sitycloud_profile', array( $this, 'render_profile_shortcode' ) );
        add_action( 'admin_menu', array( $this, 'register_admin_menu' ) );
    }

    /**
     * طباعة CSS مرّة واحدة فقط
     */
    private function print_inline_css() {
        if ( self::$css_printed ) {
            return;
        }
        self::$css_printed = true;
        ?>
        <style>
            :root {
                --sity-primary: #7c3aed;
                --sity-primary-deep: #4c1d95;
                --sity-primary-soft: #ede9fe;
                --sity-card-bg: rgba(255, 255, 255, 0.98);
                --sity-border-soft: rgba(148, 163, 184, 0.40);
                --sity-border-strong: rgba(124, 58, 237, 0.70);
                --sity-text-main: #020617;
                --sity-text-muted: #6b7280;
                --sity-shadow-soft: 0 22px 55px rgba(15, 23, 42, 0.18);
                --sity-success: #16a34a;
                --sity-error: #dc2626;
            }

            .sity-profile-wrapper {
                max-width: 980px;
                margin: 30px auto;
                font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
                color: var(--sity-text-main);
            }

            .sity-profile-header {
                display: flex;
                justify-content: space-between;
                gap: 12px;
                align-items: center;
                margin-bottom: 18px;
            }

            .sity-profile-header-title {
                font-size: 1.3rem;
                font-weight: 700;
                display: flex;
                align-items: center;
                gap: 8px;
            }

            .sity-profile-header-title span.sity-dot {
                width: 10px;
                height: 10px;
                border-radius: 999px;
                background: linear-gradient(135deg, #8b5cf6, #4f46e5);
                box-shadow: 0 0 0 4px rgba(129, 140, 248, 0.4);
            }

            .sity-profile-header-sub {
                font-size: 0.9rem;
                color: var(--sity-text-muted);
            }

            .sity-profile-grid {
                display: grid;
                grid-template-columns: minmax(0, 1.05fr) minmax(0, 1.35fr);
                gap: 18px;
            }

            .sity-profile-card {
                position: relative;
                background: var(--sity-card-bg);
                border-radius: 24px;
                border: 1px solid rgba(148, 163, 184, 0.25);
                box-shadow: var(--sity-shadow-soft);
                padding: 20px 18px 18px;
                overflow: hidden;
            }

            .sity-profile-card::before {
                content: "";
                position: absolute;
                inset: -1px;
                border-radius: inherit;
                padding: 1px;
                background: linear-gradient(135deg,
                    rgba(124, 58, 237, 0.25),
                    rgba(59, 130, 246, 0.15)
                );
                -webkit-mask:
                    linear-gradient(#000 0 0) content-box,
                    linear-gradient(#000 0 0);
                -webkit-mask-composite: xor;
                        mask-composite: exclude;
                pointer-events: none;
            }

            .sity-profile-card-header {
                display: flex;
                justify-content: space-between;
                align-items: center;
                margin-bottom: 14px;
            }

            .sity-profile-card-title {
                font-size: 1rem;
                font-weight: 600;
            }

            .sity-profile-badge {
                font-size: 0.75rem;
                padding: 3px 9px;
                border-radius: 999px;
                background: rgba(124, 58, 237, 0.09);
                color: var(--sity-primary-deep);
                border: 1px solid rgba(124, 58, 237, 0.25);
            }

            .sity-profile-summary-main {
                display: flex;
                flex-direction: column;
                gap: 5px;
                margin-bottom: 10px;
            }

            .sity-profile-user-name {
                font-weight: 700;
                font-size: 1.05rem;
            }

            .sity-profile-user-email {
                font-size: 0.9rem;
                color: var(--sity-text-muted);
                direction: ltr;
                text-align: left;
            }

            .sity-profile-chip-row {
                display: flex;
                flex-wrap: wrap;
                gap: 8px;
                margin-bottom: 10px;
            }

            .sity-chip {
                display: inline-flex;
                align-items: center;
                gap: 6px;
                padding: 5px 9px;
                border-radius: 999px;
                font-size: 0.78rem;
                background: rgba(248, 250, 252, 0.9);
                border: 1px solid rgba(148, 163, 184, 0.45);
                color: var(--sity-text-muted);
            }

            .sity-chip strong {
                color: var(--sity-text-main);
                font-weight: 600;
            }

            .sity-profile-stats {
                display: grid;
                grid-template-columns: repeat(2, minmax(0, 1fr));
                gap: 10px;
                margin-top: 6px;
            }

            .sity-stat-card {
                border-radius: 16px;
                padding: 10px 10px 9px;
                background: rgba(248, 250, 252, 0.95);
                border: 1px solid rgba(148, 163, 184, 0.4);
                font-size: 0.8rem;
            }

            .sity-stat-label {
                color: var(--sity-text-muted);
                margin-bottom: 2px;
            }

            .sity-stat-value {
                font-weight: 700;
                font-size: 0.95rem;
                color: var(--sity-primary-deep);
            }

            .sity-stat-sub {
                font-size: 0.78rem;
                color: var(--sity-text-muted);
            }

            .sity-profile-form {
                margin-top: 4px;
            }

            .sity-field-group {
                display: flex;
                flex-direction: column;
                gap: 4px;
                margin-bottom: 9px;
            }

            .sity-label {
                font-size: 0.8rem;
                font-weight: 500;
                color: var(--sity-text-muted);
                display: flex;
                justify-content: space-between;
            }

            .sity-label span.sity-required {
                color: var(--sity-primary);
                font-weight: 700;
            }

            .sity-input,
            .sity-textarea {
                width: 100%;
                border-radius: 14px;
                border: 1px solid var(--sity-border-soft);
                background: rgba(255, 255, 255, 0.98);
                padding: 9px 10px;
                font-size: 0.9rem;
                color: var(--sity-text-main);
                outline: none;
                transition:
                    border-color 0.18s ease,
                    box-shadow 0.18s ease,
                    transform 0.08s ease,
                    background 0.18s ease;
            }

            .sity-input:focus,
            .sity-textarea:focus {
                border-color: var(--sity-border-strong);
                box-shadow:
                    0 0 0 1px rgba(124, 58, 237, 0.10),
                    0 0 0 6px rgba(124, 58, 237, 0.18);
                transform: translateY(-1px);
            }

            .sity-input[readonly],
            .sity-input[disabled] {
                background: rgba(248, 250, 252, 0.9);
                color: var(--sity-text-muted);
                cursor: not-allowed;
            }

            .sity-textarea {
                min-height: 70px;
                resize: vertical;
            }

            .sity-profile-actions {
                display: flex;
                justify-content: flex-end;
                margin-top: 10px;
            }

            .sity-btn-primary {
                border-radius: 999px;
                border: none;
                padding: 10px 18px;
                font-size: 0.9rem;
                font-weight: 600;
                cursor: pointer;
                background: linear-gradient(135deg, #8b5cf6, #4f46e5);
                color: #ffffff;
                box-shadow: 0 15px 32px rgba(55, 48, 163, 0.45);
                transition:
                    transform 0.12s ease,
                    box-shadow 0.12s ease,
                    filter 0.12s ease;
            }

            .sity-btn-primary:hover {
                transform: translateY(-1px);
                filter: brightness(1.05);
                box-shadow: 0 18px 38px rgba(55, 48, 163, 0.50);
            }

            .sity-btn-primary:active {
                transform: translateY(0);
                box-shadow: 0 10px 24px rgba(55, 48, 163, 0.35);
            }

            .sity-profile-message {
                margin-bottom: 12px;
                border-radius: 14px;
                padding: 9px 11px;
                font-size: 0.85rem;
                border: 1px solid transparent;
            }

            .sity-profile-message.sity-success {
                border-color: rgba(22, 163, 74, 0.25);
                background: rgba(240, 253, 250, 0.98);
                color: #166534;
            }

            .sity-profile-message.sity-error {
                border-color: rgba(220, 38, 38, 0.25);
                background: rgba(254, 242, 242, 0.98);
                color: #b91c1c;
            }

            .sity-profile-empty {
                padding: 18px 16px;
                border-radius: 18px;
                background: rgba(248, 250, 252, 0.9);
                border: 1px dashed rgba(148, 163, 184, 0.6);
                font-size: 0.9rem;
                color: var(--sity-text-muted);
            }

            .sity-profile-empty a {
                color: var(--sity-primary);
                text-decoration: none;
                font-weight: 600;
            }

            .sity-profile-empty a:hover {
                text-decoration: underline;
            }

            @media (max-width: 820px) {
                .sity-profile-grid {
                    grid-template-columns: minmax(0, 1fr);
                }
            }

            @media (max-width: 600px) {
                .sity-profile-wrapper {
                    padding-inline: 10px;
                }

                .sity-profile-header {
                    flex-direction: column;
                    align-items: flex-start;
                    gap: 4px;
                }
            }

            /* جدول الأدمن */
            .sity-admin-table-wrapper {
                margin-top: 20px;
            }

            .sity-admin-table-wrapper table td,
            .sity-admin-table-wrapper table th {
                font-size: 12px;
            }

        </style>
        <?php
    }

    /**
     * جلب رصيد المحفظة من البلجن SityCloud Wallet لو موجود
     */
    private function get_wallet_balance( $user_id ) {
        if ( class_exists( 'SityCloud_Wallet_Plugin' ) ) {
            // البلجن فيه get_instance() + get_user_balance
            $plugin = SityCloud_Wallet_Plugin::get_instance();
            if ( method_exists( $plugin, 'get_user_balance' ) ) {
                return $plugin->get_user_balance( $user_id );
            }
        }

        // fallback لو البلجن مش شغال – نقرأ الميتا مباشرة
        $balance = get_user_meta( $user_id, '_sitycloud_wallet_balance', true );
        if ( $balance === '' || $balance === null ) {
            $balance = 0;
        }
        return floatval( $balance );
    }

    /**
     * عدد المواقع – نستخدم fawry_get_user_sites_count لو متاحة
     */
    private function get_sites_count( $user_id ) {
        if ( function_exists( 'fawry_get_user_sites_count' ) ) {
            return intval( fawry_get_user_sites_count( $user_id ) );
        }

        if ( is_multisite() ) {
            $sites = get_blogs_of_user( $user_id );
            return is_array( $sites ) ? count( $sites ) : 0;
        }

        return 0;
    }

    /**
     * معالجة فورم البروفايل
     */
    private function handle_form_submit( $user_id ) {
        if ( ! isset( $_POST['sitycloud_profile_update'] ) ) {
            return array( 'status' => '', 'message' => '' );
        }

        if (
            ! isset( $_POST['sitycloud_profile_nonce'] ) ||
            ! wp_verify_nonce( $_POST['sitycloud_profile_nonce'], 'sitycloud_profile_update' )
        ) {
            return array(
                'status'  => 'error',
                'message' => 'حدث خطأ في التحقق من الطلب، برجاء إعادة المحاولة.'
            );
        }

        if ( ! is_user_logged_in() || get_current_user_id() !== $user_id ) {
            return array(
                'status'  => 'error',
                'message' => 'لا تملك صلاحية تعديل هذه البيانات.'
            );
        }

        // الحقول المسموح تعديلها – مربوطة بمفاتيح fawry
        $fields = array(
            'fawry_client_type'           => 'fawry_client_type',
            'fawry_company_name'          => 'fawry_company_name',
            'fawry_phone_number'         => 'fawry_phone_number',
            'fawry_country'              => 'fawry_country',
            'fawry_city'                 => 'fawry_city',
            'fawry_whatsapp_link'        => 'fawry_whatsapp_link',
            'fawry_industry'             => 'fawry_industry',
            'fawry_activity_type'        => 'fawry_activity_type',
            'fawry_referral_source'      => 'fawry_referral_source',
            'fawry_points_payment_method'=> 'fawry_points_payment_method',
            'fawry_project_description'  => 'fawry_project_description',
            'fawry_marketing_budget'     => 'fawry_marketing_budget',
            'fawry_products_description' => 'fawry_products_description',
            'fawry_website_goal'         => 'fawry_website_goal',
            'fawry_site_url'             => 'fawry_site_url',
        );

        foreach ( $fields as $field_name => $meta_key ) {
            if ( isset( $_POST[ $field_name ] ) ) {
                $value = $_POST[ $field_name ];

                if ( $meta_key === 'fawry_site_url' || $meta_key === 'fawry_whatsapp_link' ) {
                    $value = esc_url_raw( $value );
                } else {
                    $value = sanitize_text_field( $value );
                }

                update_user_meta( $user_id, $meta_key, $value );
            }
        }

        return array(
            'status'  => 'success',
            'message' => 'تم تحديث بيانات حسابك بنجاح ✅'
        );
    }

    /**
     * شورت كود البروفايل
     */
    public function render_profile_shortcode( $atts ) {
        if ( ! is_user_logged_in() ) {
            $this->print_inline_css();
            ob_start();
            ?>
            <div class="sity-profile-wrapper">
                <div class="sity-profile-empty">
                    يجب تسجيل الدخول لعرض لوحة حسابك.  
                    <a href="<?php echo esc_url( wp_login_url( get_permalink() ) ); ?>">تسجيل الدخول</a>
                </div>
            </div>
            <?php
            return ob_get_clean();
        }

        $this->print_inline_css();

        $user    = wp_get_current_user();
        $user_id = $user->ID;

        // معالجة الفورم (لو فيه POST)
        $result = $this->handle_form_submit( $user_id );

        // بيانات أساسية من البلجن fawry + wallet
        $iban         = get_user_meta( $user_id, 'fawry_iban', true );
        $points       = intval( get_user_meta( $user_id, 'fawry_points', true ) ?: 0 );
        $sites_count  = $this->get_sites_count( $user_id );
        $wallet       = $this->get_wallet_balance( $user_id );
        $member_since = $user->user_registered ? mysql2date( 'Y/m/d', $user->user_registered ) : '';

        // بيانات العميل التفصيلية من fawry_*
        $client_type           = get_user_meta( $user_id, 'fawry_client_type', true );
        $company_name          = get_user_meta( $user_id, 'fawry_company_name', true );
        $phone_number          = get_user_meta( $user_id, 'fawry_phone_number', true );
        $country               = get_user_meta( $user_id, 'fawry_country', true );
        $city                  = get_user_meta( $user_id, 'fawry_city', true );
        $whatsapp_link         = get_user_meta( $user_id, 'fawry_whatsapp_link', true );
        $industry              = get_user_meta( $user_id, 'fawry_industry', true );
        $activity_type         = get_user_meta( $user_id, 'fawry_activity_type', true );
        $referral_source       = get_user_meta( $user_id, 'fawry_referral_source', true );
        $points_payment_method = get_user_meta( $user_id, 'fawry_points_payment_method', true );
        $project_description   = get_user_meta( $user_id, 'fawry_project_description', true );
        $marketing_budget      = get_user_meta( $user_id, 'fawry_marketing_budget', true );
        $products_description  = get_user_meta( $user_id, 'fawry_products_description', true );
        $website_goal          = get_user_meta( $user_id, 'fawry_website_goal', true );
        $site_url              = get_user_meta( $user_id, 'fawry_site_url', true );

        ob_start();
        ?>
        <div class="sity-profile-wrapper">
            <div class="sity-profile-header">
                <div>
                    <div class="sity-profile-header-title">
                        <span class="sity-dot"></span>
                        لوحة حسابك في Sity Cloud
                    </div>
                    <div class="sity-profile-header-sub">
                        راجع بياناتك الأساسية، وحدّث معلومات شركتك ومشروعك في أي وقت.
                    </div>
                </div>
            </div>

            <?php if ( ! empty( $result['status'] ) ) : ?>
                <div class="sity-profile-message <?php echo $result['status'] === 'success' ? 'sity-success' : 'sity-error'; ?>">
                    <?php echo esc_html( $result['message'] ); ?>
                </div>
            <?php endif; ?>

            <div class="sity-profile-grid">

                <!-- كارت ملخص الحساب -->
                <div class="sity-profile-card">
                    <div class="sity-profile-card-header">
                        <div class="sity-profile-card-title">ملخص حسابك</div>
                        <div class="sity-profile-badge">عميل Sity Cloud</div>
                    </div>

                    <div class="sity-profile-summary-main">
                        <div class="sity-profile-user-name">
                            <?php echo esc_html( $user->display_name ?: $user->user_login ); ?>
                        </div>
                        <div class="sity-profile-user-email">
                            <?php echo esc_html( $user->user_email ); ?>
                        </div>
                    </div>

                    <div class="sity-profile-chip-row">
                        <div class="sity-chip">
                            <strong>IBAN</strong>
                            <span>
                                <?php
                                if ( ! empty( $iban ) ) {
                                    $masked = $iban;
                                    if ( strlen( $iban ) > 8 ) {
                                        $masked = substr( $iban, 0, 4 ) . ' •••• •••• ' . substr( $iban, -4 );
                                    }
                                    echo esc_html( $masked );
                                } else {
                                    echo 'لم يتم توليد IBAN بعد';
                                }
                                ?>
                            </span>
                        </div>

                        <div class="sity-chip">
                            <strong>الحالة</strong>
                            <span>حساب فعّال</span>
                        </div>

                        <?php if ( $client_type ) : ?>
                            <div class="sity-chip">
                                <strong>نوع العميل</strong>
                                <span><?php echo esc_html( $client_type ); ?></span>
                            </div>
                        <?php endif; ?>
                    </div>

                    <div class="sity-profile-stats">
                        <div class="sity-stat-card">
                            <div class="sity-stat-label">رصيد المحفظة</div>
                            <div class="sity-stat-value">
                                <?php echo esc_html( number_format( $wallet, 2 ) ); ?> ج.م
                            </div>
                            <div class="sity-stat-sub">يمكنك استخدامه في اشتراكاتك وخدماتك.</div>
                        </div>

                        <div class="sity-stat-card">
                            <div class="sity-stat-label">عدد المواقع</div>
                            <div class="sity-stat-value">
                                <?php echo esc_html( $sites_count ); ?>
                            </div>
                            <div class="sity-stat-sub">مواقع تم إنشاؤها عبر Sity Cloud.</div>
                        </div>

                        <div class="sity-stat-card">
                            <div class="sity-stat-label">نقاطك الحالية</div>
                            <div class="sity-stat-value">
                                <?php echo esc_html( $points ); ?>
                            </div>
                            <div class="sity-stat-sub">مرتبطة بنظام النقاط في ستي كلاود.</div>
                        </div>

                        <?php if ( $member_since ) : ?>
                            <div class="sity-stat-card">
                                <div class="sity-stat-label">عضو منذ</div>
                                <div class="sity-stat-value">
                                    <?php echo esc_html( $member_since ); ?>
                                </div>
                                <div class="sity-stat-sub">تاريخ إنشاء حسابك.</div>
                            </div>
                        <?php endif; ?>
                    </div>

                    <?php if ( $company_name ) : ?>
                        <div style="margin-top:10px;font-size:0.8rem;color:var(--sity-text-muted);">
                            <strong>اسم الشركة / البراند:</strong>
                            <span><?php echo esc_html( $company_name ); ?></span>
                        </div>
                    <?php endif; ?>
                </div>

                <!-- كارت تعديل البيانات -->
                <div class="sity-profile-card">
                    <div class="sity-profile-card-header">
                        <div class="sity-profile-card-title">بيانات حسابك الأساسية</div>
                        <div class="sity-profile-badge">تعديل البيانات</div>
                    </div>

                    <form method="post" class="sity-profile-form">
                        <?php wp_nonce_field( 'sitycloud_profile_update', 'sitycloud_profile_nonce' ); ?>
                        <input type="hidden" name="sitycloud_profile_update" value="1" />

                        <!-- الاسم و الإيميل و IBAN (قراءة فقط) -->
                        <div class="sity-field-group">
                            <label class="sity-label">
                                <span>الاسم الكامل</span>
                            </label>
                            <input type="text" class="sity-input" value="<?php echo esc_attr( $user->display_name ?: $user->user_login ); ?>" readonly />
                        </div>

                        <div class="sity-field-group">
                            <label class="sity-label">
                                <span>البريد الإلكتروني</span>
                            </label>
                            <input type="email" class="sity-input" value="<?php echo esc_attr( $user->user_email ); ?>" readonly />
                        </div>

                        <div class="sity-field-group">
                            <label class="sity-label">
                                <span>IBAN</span>
                            </label>
                            <input type="text" class="sity-input" value="<?php echo esc_attr( $iban ); ?>" readonly />
                        </div>

                        <hr style="margin:10px 0 12px;border:none;border-top:1px dashed rgba(148,163,184,0.6);" />

                        <!-- بيانات قابلة للتعديل (fawry_*) -->
                        <div class="sity-field-group">
                            <label class="sity-label">
                                <span>نوع العميل</span>
                            </label>
                            <input type="text" name="fawry_client_type" class="sity-input"
                                   value="<?php echo esc_attr( $client_type ); ?>"
                                   placeholder="فرد / شركة / مؤسسة..." />
                        </div>

                        <div class="sity-field-group">
                            <label class="sity-label">
                                <span>اسم الشركة أو البراند</span>
                            </label>
                            <input type="text" name="fawry_company_name" class="sity-input"
                                   value="<?php echo esc_attr( $company_name ); ?>"
                                   placeholder="اسم مشروعك أو شركتك" />
                        </div>

                        <div class="sity-field-group">
                            <label class="sity-label">
                                <span>رقم الجوال</span>
                                <span class="sity-required">*</span>
                            </label>
                            <input type="text" name="fawry_phone_number" class="sity-input"
                                   value="<?php echo esc_attr( $phone_number ); ?>"
                                   placeholder="مثال: 01000000000" />
                        </div>

                        <div class="sity-field-group">
                            <label class="sity-label">
                                <span>رابط واتساب</span>
                            </label>
                            <input type="url" name="fawry_whatsapp_link" class="sity-input"
                                   value="<?php echo esc_attr( $whatsapp_link ); ?>"
                                   placeholder="رابط واتساب مباشر (اختياري)" />
                        </div>

                        <div class="sity-field-group">
                            <label class="sity-label">
                                <span>الدولة</span>
                            </label>
                            <input type="text" name="fawry_country" class="sity-input"
                                   value="<?php echo esc_attr( $country ); ?>"
                                   placeholder="مثال: مصر، السعودية..." />
                        </div>

                        <div class="sity-field-group">
                            <label class="sity-label">
                                <span>المدينة</span>
                            </label>
                            <input type="text" name="fawry_city" class="sity-input"
                                   value="<?php echo esc_attr( $city ); ?>"
                                   placeholder="مثال: القاهرة، الرياض..." />
                        </div>

                        <div class="sity-field-group">
                            <label class="sity-label">
                                <span>مجال العمل</span>
                            </label>
                            <input type="text" name="fawry_industry" class="sity-input"
                                   value="<?php echo esc_attr( $industry ); ?>"
                                   placeholder="مجال نشاطك العام (تجارة إلكترونية، تعليم، طب...)" />
                        </div>

                        <div class="sity-field-group">
                            <label class="sity-label">
                                <span>نوع النشاط</span>
                            </label>
                            <input type="text" name="fawry_activity_type" class="sity-input"
                                   value="<?php echo esc_attr( $activity_type ); ?>"
                                   placeholder="مثال: متجر أحذية، مركز تدريبي، عيادة..." />
                        </div>

                        <div class="sity-field-group">
                            <label class="sity-label">
                                <span>طريقة التعرف على Sity Cloud</span>
                            </label>
                            <input type="text" name="fawry_referral_source" class="sity-input"
                                   value="<?php echo esc_attr( $referral_source ); ?>"
                                   placeholder="صديق، جوجل، إعلان، سوشيال ميديا..." />
                        </div>

                        <div class="sity-field-group">
                            <label class="sity-label">
                                <span>آلية دفع النقاط</span>
                            </label>
                            <input type="text" name="fawry_points_payment_method" class="sity-input"
                                   value="<?php echo esc_attr( $points_payment_method ); ?>"
                                   placeholder="أونلاين، تحويل بنكي، محفظة..." />
                        </div>

                        <div class="sity-field-group">
                            <label class="sity-label">
                                <span>وصف المشروع</span>
                            </label>
                            <textarea name="fawry_project_description" class="sity-textarea"
                                      placeholder="احكي لنا عن مشروعك وطبيعة نشاطك."><?php echo esc_textarea( $project_description ); ?></textarea>
                        </div>

                        <div class="sity-field-group">
                            <label class="sity-label">
                                <span>ميزانية التسويق التقريبية</span>
                            </label>
                            <input type="text" name="fawry_marketing_budget" class="sity-input"
                                   value="<?php echo esc_attr( $marketing_budget ); ?>"
                                   placeholder="مثال: 2000 ج.م شهريًا" />
                        </div>

                        <div class="sity-field-group">
                            <label class="sity-label">
                                <span>وصف المنتجات / الخدمات</span>
                            </label>
                            <textarea name="fawry_products_description" class="sity-textarea"
                                      placeholder="ما هي المنتجات أو الخدمات التي تقدمها؟"><?php echo esc_textarea( $products_description ); ?></textarea>
                        </div>

                        <div class="sity-field-group">
                            <label class="sity-label">
                                <span>هدف الموقع</span>
                            </label>
                            <input type="text" name="fawry_website_goal" class="sity-input"
                                   value="<?php echo esc_attr( $website_goal ); ?>"
                                   placeholder="بيع أونلاين، حجز مواعيد، عرض بورتفوليو..." />
                        </div>

                        <div class="sity-field-group">
                            <label class="sity-label">
                                <span>رابط موقعك (إن وجد)</span>
                            </label>
                            <input type="url" name="fawry_site_url" class="sity-input"
                                   value="<?php echo esc_attr( $site_url ); ?>"
                                   placeholder="https://example.com" />
                        </div>

                        <div class="sity-profile-actions">
                            <button type="submit" class="sity-btn-primary">
                                حفظ وتحديث البيانات
                            </button>
                        </div>
                    </form>
                </div>

            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * صفحة الأدمن – قائمة بكل العملاء
     */
    public function register_admin_menu() {
        add_menu_page(
            'عملاء Sity Cloud',
            'عملاء Sity Cloud',
            'manage_options',
            'sitycloud_clients',
            array( $this, 'render_admin_clients_page' ),
            'dashicons-admin-users',
            26
        );
    }

    public function render_admin_clients_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        $this->print_inline_css();

        $users = get_users();

        echo '<div class="wrap">';
        echo '<h1>بيانات عملاء Sity Cloud</h1>';

        if ( ! $users ) {
            echo '<p>لا توجد بيانات عملاء حتى الآن.</p>';
            echo '</div>';
            return;
        }

        echo '<div class="sity-admin-table-wrapper">';
        echo '<table class="wp-list-table widefat fixed striped">';
        echo '<thead><tr>';
        echo '<th>الاسم</th>';
        echo '<th>الإيميل</th>';
        echo '<th>رقم الجوال</th>';
        echo '<th>الشركة / البراند</th>';
        echo '<th>الدولة</th>';
        echo '<th>المدينة</th>';
        echo '<th>رصيد المحفظة</th>';
        echo '<th>النقاط</th>';
        echo '<th>عدد المواقع</th>';
        echo '<th>IBAN</th>';
        echo '</tr></thead><tbody>';

        foreach ( $users as $user ) {
            $user_id      = $user->ID;
            $phone        = get_user_meta( $user_id, 'fawry_phone_number', true );
            $company      = get_user_meta( $user_id, 'fawry_company_name', true );
            $country      = get_user_meta( $user_id, 'fawry_country', true );
            $city         = get_user_meta( $user_id, 'fawry_city', true );
            $points       = intval( get_user_meta( $user_id, 'fawry_points', true ) ?: 0 );
            $wallet       = $this->get_wallet_balance( $user_id );
            $sites_count  = $this->get_sites_count( $user_id );
            $iban         = get_user_meta( $user_id, 'fawry_iban', true );

            echo '<tr>';
            echo '<td>' . esc_html( $user->display_name ) . '</td>';
            echo '<td>' . esc_html( $user->user_email ) . '</td>';
            echo '<td>' . esc_html( $phone ) . '</td>';
            echo '<td>' . esc_html( $company ) . '</td>';
            echo '<td>' . esc_html( $country ) . '</td>';
            echo '<td>' . esc_html( $city ) . '</td>';
            echo '<td>' . esc_html( number_format( $wallet, 2 ) ) . ' ج.م</td>';
            echo '<td>' . esc_html( $points ) . '</td>';
            echo '<td>' . esc_html( $sites_count ) . '</td>';
            echo '<td>' . esc_html( $iban ) . '</td>';
            echo '</tr>';
        }

        echo '</tbody></table>';
        echo '</div>'; // wrapper

        echo '</div>'; // .wrap
    }
}

SityCloud_Profile_Bridge::get_instance();
