<?php
/**
 * Plugin Name: Sity Cloud Network Manager
 * Plugin URI:  https://sity.cloud
 * Description: لوحة تحكم لشبكة ووردبريس مالتي سايت تعرض كل المواقع في شكل كروت مع بيانات العميل والإضافات وآخر تسجيل دخول.
 * Author:      Sity Cloud
 * Author URI:  https://sity.cloud
 * Version:     1.0.0
 * Network:     true
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // لا تسمح بالدخول المباشر
}

/**
 * تسجيل آخر تسجيل دخول لكل مستخدم
 */
function sitycloud_record_user_login( $user_login, $user ) {
    if ( ! $user instanceof WP_User ) {
        $user = get_user_by( 'login', $user_login );
        if ( ! $user ) {
            return;
        }
    }

    // تخزين آخر تسجيل دخول في user_meta
    update_user_meta(
        $user->ID,
        'sitycloud_last_login',
        current_time( 'timestamp' ) // توقيت السيرفر حسب إعداد ووردبريس
    );
}
add_action( 'wp_login', 'sitycloud_record_user_login', 10, 2 );

/**
 * إضافة صفحة في Network Admin
 */
function sitycloud_add_network_admin_menu() {
    if ( ! is_multisite() ) {
        return;
    }

    if ( ! current_user_can( 'manage_network' ) ) {
        return;
    }

    add_menu_page(
        __( 'Sity Cloud Network', 'sitycloud' ),
        __( 'Sity Cloud', 'sitycloud' ),
        'manage_network',
        'sitycloud-network-dashboard',
        'sitycloud_render_network_dashboard',
        'dashicons-cloud',
        3
    );
}
add_action( 'network_admin_menu', 'sitycloud_add_network_admin_menu' );

/**
 * إنشاء رابط Screenshot باستخدام mShots
 *
 * @param string $url
 * @return string
 */
function sitycloud_get_site_screenshot_url( $url ) {
    $url = trim( $url );
    if ( empty( $url ) ) {
        return '';
    }

    // ترميز الرابط لاستخدامه في mShots
    $encoded = rawurlencode( $url );

    // mShots من WordPress.com - صورة شاشة للموقع
    $screenshot_url = 'https://s0.wordpress.com/mshots/v1/' . $encoded . '?w=800&h=450';

    return esc_url( $screenshot_url );
}

/**
 * تنسيق وقت آخر تسجيل دخول
 *
 * @param int|string $timestamp
 * @return string
 */
function sitycloud_format_last_login( $timestamp ) {
    if ( empty( $timestamp ) ) {
        return 'لا يوجد تسجيل دخول مسجَّل بعد';
    }

    $timestamp = (int) $timestamp;

    $date_format = get_option( 'date_format', 'Y-m-d' );
    $time_format = get_option( 'time_format', 'H:i' );

    $formatted = date_i18n( $date_format . ' ' . $time_format, $timestamp );

    // مثال: "منذ 3 ساعات (2025-11-19 12:30)"
    $human_diff = human_time_diff( $timestamp, current_time( 'timestamp' ) );

    return sprintf(
        '%s (%s %s)',
        esc_html( $formatted ),
        esc_html__( 'منذ', 'sitycloud' ),
        esc_html( $human_diff )
    );
}

/**
 * تنسيق وقت آخر تعديل للموقع
 *
 * @param string $mysql_datetime
 * @return string
 */
function sitycloud_format_last_updated( $mysql_datetime ) {
    if ( empty( $mysql_datetime ) || $mysql_datetime === '0000-00-00 00:00:00' ) {
        return 'غير محدد';
    }

    $timestamp = strtotime( $mysql_datetime );

    if ( ! $timestamp ) {
        return esc_html( $mysql_datetime );
    }

    $date_format = get_option( 'date_format', 'Y-m-d' );
    $time_format = get_option( 'time_format', 'H:i' );

    $formatted = date_i18n( $date_format . ' ' . $time_format, $timestamp );
    $human_diff = human_time_diff( $timestamp, current_time( 'timestamp' ) );

    return sprintf(
        '%s (%s %s)',
        esc_html( $formatted ),
        esc_html__( 'منذ', 'sitycloud' ),
        esc_html( $human_diff )
    );
}

/**
 * صفحة لوحة تحكم الشبكة
 */
function sitycloud_render_network_dashboard() {
    if ( ! current_user_can( 'manage_network' ) ) {
        wp_die( __( 'ليس لديك صلاحية الوصول لهذه الصفحة.', 'sitycloud' ) );
    }

    if ( ! is_multisite() ) {
        echo '<div class="wrap"><h1>Sity Cloud Network</h1><p>يجب تفعيل وضع Multisite لاستخدام هذه الإضافة.</p></div>';
        return;
    }

    // جلب كل المواقع في الشبكة
    $sites = get_sites( array(
        'number' => 0, // 0 = كل المواقع
        'orderby' => 'id',
        'order'   => 'ASC',
    ) );

    ?>
    <div class="wrap sitycloud-network-wrap">
        <h1 style="margin-bottom: 16px;">Sity Cloud – Network Manager</h1>
        <p style="max-width: 720px; font-size: 14px; color: #555;">
            هذه اللوحة تعرض جميع المواقع في شبكة Sity Cloud Multisite في شكل كروت. 
            لكل موقع يمكنك رؤية اسم الموقع، الرابط، بيانات العميل (أول Admin)، 
            عدد الإضافات المفعّلة، آخر تسجيل دخول، وآخر وقت تعديل للموقع.
        </p>

        <!-- تحميل Bootstrap من CDN (بما أن البلجن ملف واحد فقط) -->
        <link rel="stylesheet"
              href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css"
              integrity="sha384-QWTKZyjpPEjISv5WaRU9OFeRpok6YctnYmDr5pNlyT2bRjXh0JMhjY6hW+ALEwIH"
              crossorigin="anonymous">

        <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"
                integrity="sha384-YvpcrYf0tY3lHB60NNkmXc5s9fDVZLESaAA55NDzOxhy9GkcIdslK1eN7N6jIeHz"
                crossorigin="anonymous"></script>

        <style>
            .sitycloud-network-wrap {
                direction: rtl; /* لأن اللغة عربية */
            }

            .sitycloud-sites-grid {
                margin-top: 24px;
            }

            .sitycloud-site-card {
                border-radius: 18px;
                overflow: hidden;
                box-shadow: 0 12px 30px rgba(0,0,0,0.08);
                border: 1px solid rgba(120, 120, 255, 0.15);
                background: linear-gradient(135deg, rgba(98, 0, 255, 0.07), rgba(255, 255, 255, 0.95));
                transition: transform 0.2s ease, box-shadow 0.2s ease, border-color 0.2s ease;
                position: relative;
            }

            .sitycloud-site-card:hover {
                transform: translateY(-4px);
                box-shadow: 0 18px 40px rgba(0,0,0,0.12);
                border-color: rgba(98, 0, 255, 0.4);
            }

            .sitycloud-site-badge {
                position: absolute;
                top: 10px;
                left: 10px;
                padding: 4px 12px;
                border-radius: 999px;
                background: rgba(0,0,0,0.65);
                color: #fff;
                font-size: 11px;
            }

            .sitycloud-site-img-wrapper {
                max-height: 220px;
                overflow: hidden;
                background: #050220;
            }

            .sitycloud-site-img-wrapper img {
                width: 100%;
                object-fit: cover;
                display: block;
                filter: saturate(1.15);
            }

            .sitycloud-site-card .card-body {
                padding: 1rem 1.25rem 1.15rem;
            }

            .sitycloud-site-name {
                font-weight: 700;
                font-size: 1.05rem;
                margin-bottom: 4px;
                display: flex;
                align-items: center;
                gap: 8px;
            }

            .sitycloud-site-name span.badge {
                font-size: 10px;
                background: rgba(98, 0, 255, 0.12);
                color: #3b0b9c;
                border-radius: 999px;
            }

            .sitycloud-site-url a {
                font-size: 13px;
                color: #4b3aff;
                text-decoration: none;
                word-break: break-all;
            }

            .sitycloud-site-url a:hover {
                text-decoration: underline;
            }

            .sitycloud-meta-row {
                display: grid;
                grid-template-columns: repeat(2, minmax(0, 1fr));
                gap: 6px 16px;
                margin-top: 10px;
                font-size: 12px;
            }

            .sitycloud-meta-label {
                font-weight: 600;
                color: #333;
            }

            .sitycloud-meta-value {
                color: #555;
            }

            .sitycloud-pills {
                display: flex;
                flex-wrap: wrap;
                gap: 6px;
                margin-top: 10px;
            }

            .sitycloud-pill {
                font-size: 11px;
                padding: 4px 10px;
                border-radius: 999px;
                background: rgba(255,255,255,0.7);
                border: 1px solid rgba(120,120,255,0.35);
                color: #26204b;
            }

            .sitycloud-client-name {
                font-weight: 600;
            }

            .sitycloud-footer-actions {
                display: flex;
                justify-content: space-between;
                align-items: center;
                margin-top: 12px;
                gap: 8px;
            }

            .sitycloud-btn {
                border-radius: 999px;
                padding: 6px 14px;
                font-size: 12px;
                border: none;
                display: inline-flex;
                align-items: center;
                gap: 4px;
            }

            .sitycloud-btn-primary {
                background: linear-gradient(135deg, #7a3bff, #b245ff);
                color: #fff;
            }

            .sitycloud-btn-outline {
                background: transparent;
                color: #3e2baa;
                border: 1px solid rgba(62, 43, 170, 0.35);
            }

            @media (max-width: 767px) {
                .sitycloud-meta-row {
                    grid-template-columns: 1fr;
                }
            }
        </style>

        <div class="sitycloud-sites-grid">
            <div class="row g-4">
                <?php
                if ( empty( $sites ) ) :
                    ?>
                    <div class="col-12">
                        <div class="alert alert-warning">
                            لا توجد مواقع مسجَّلة في هذه الشبكة حتى الآن.
                        </div>
                    </div>
                    <?php
                else :
                    foreach ( $sites as $site ) :

                        /** @var WP_Site $site */
                        $blog_id      = (int) $site->blog_id;
                        $blog_domain  = $site->domain;
                        $blog_path    = $site->path;
                        $last_updated = $site->last_updated;

                        // تفاصيل إضافية من داخل الموقع
                        switch_to_blog( $blog_id );

                        $blogname        = get_option( 'blogname' );
                        $home_url        = get_option( 'home' );
                        $site_url        = get_option( 'siteurl' );
                        $active_plugins  = (array) get_option( 'active_plugins', array() );
                        $active_count    = count( $active_plugins );

                        // أول مسؤول في الموقع نعتبره "العميل"
                        $client_user     = null;
                        $client_name     = '';
                        $client_email    = '';
                        $client_lastlogin = '';

                        $admins = get_users( array(
                            'role'   => 'administrator',
                            'number' => 1,
                            'fields' => array( 'ID', 'display_name', 'user_email' ),
                        ) );

                        if ( ! empty( $admins ) ) {
                            $client_user  = $admins[0];
                            $client_name  = $client_user->display_name;
                            $client_email = $client_user->user_email;

                            $raw_last_login = get_user_meta( $client_user->ID, 'sitycloud_last_login', true );
                            $client_lastlogin = sitycloud_format_last_login( $raw_last_login );
                        } else {
                            $client_name      = 'لا يوجد Admin محدد';
                            $client_email     = '-';
                            $client_lastlogin = 'غير متوفر';
                        }

                        restore_current_blog();

                        // رابط Screenshot
                        $screenshot_url = sitycloud_get_site_screenshot_url( $home_url ? $home_url : $site_url );

                        ?>
                        <div class="col-xl-4 col-lg-6 col-md-6 col-sm-12">
                            <div class="card sitycloud-site-card h-100">
                                <div class="sitycloud-site-badge">
                                    موقع رقم #<?php echo esc_html( $blog_id ); ?>
                                </div>

                                <div class="sitycloud-site-img-wrapper">
                                    <?php if ( $screenshot_url ) : ?>
                                        <img src="<?php echo esc_url( $screenshot_url ); ?>"
                                             alt="<?php echo esc_attr( $blogname ); ?>">
                                    <?php else : ?>
                                        <div style="padding:40px; color:#fff; text-align:center; font-size:13px;">
                                            لا يمكن تحميل صورة الموقع حاليًا
                                        </div>
                                    <?php endif; ?>
                                </div>

                                <div class="card-body">
                                    <div class="sitycloud-site-name">
                                        <span><?php echo esc_html( $blogname ); ?></span>
                                        <span class="badge">
                                            <?php echo esc_html( $blog_domain . $blog_path ); ?>
                                        </span>
                                    </div>

                                    <div class="sitycloud-site-url">
                                        <?php if ( $home_url ) : ?>
                                            <a href="<?php echo esc_url( $home_url ); ?>" target="_blank" rel="noopener">
                                                <?php echo esc_html( $home_url ); ?>
                                            </a>
                                        <?php else : ?>
                                            <span>-</span>
                                        <?php endif; ?>
                                    </div>

                                    <div class="sitycloud-meta-row">
                                        <div>
                                            <div class="sitycloud-meta-label">اسم العميل</div>
                                            <div class="sitycloud-meta-value sitycloud-client-name">
                                                <?php echo esc_html( $client_name ); ?>
                                            </div>
                                        </div>

                                        <div>
                                            <div class="sitycloud-meta-label">إيميل العميل</div>
                                            <div class="sitycloud-meta-value">
                                                <?php echo esc_html( $client_email ); ?>
                                            </div>
                                        </div>

                                        <div>
                                            <div class="sitycloud-meta-label">عدد الإضافات المفعَّلة</div>
                                            <div class="sitycloud-meta-value">
                                                <?php echo intval( $active_count ); ?>
                                            </div>
                                        </div>

                                        <div>
                                            <div class="sitycloud-meta-label">آخر تعديل للموقع</div>
                                            <div class="sitycloud-meta-value">
                                                <?php echo esc_html( sitycloud_format_last_updated( $last_updated ) ); ?>
                                            </div>
                                        </div>
                                    </div>

                                    <div class="sitycloud-pills">
                                        <div class="sitycloud-pill">
                                            ID: <?php echo intval( $blog_id ); ?>
                                        </div>
                                        <div class="sitycloud-pill">
                                            Domain: <?php echo esc_html( $blog_domain ); ?>
                                        </div>
                                        <div class="sitycloud-pill">
                                            Path: <?php echo esc_html( $blog_path ); ?>
                                        </div>
                                    </div>

                                    <div class="sitycloud-meta-row" style="margin-top: 6px;">
                                        <div style="grid-column: 1 / -1;">
                                            <div class="sitycloud-meta-label">آخر تسجيل دخول للعميل</div>
                                            <div class="sitycloud-meta-value">
                                                <?php echo esc_html( $client_lastlogin ); ?>
                                            </div>
                                        </div>
                                    </div>

                                    <div class="sitycloud-footer-actions">
                                        <button type="button"
                                                class="sitycloud-btn sitycloud-btn-primary"
                                                onclick="window.open('<?php echo esc_url( network_admin_url( 'site-info.php?id=' . $blog_id ) ); ?>','_blank');">
                                            إدارة من لوحة الشبكة
                                        </button>

                                        <?php if ( $home_url ) : ?>
                                            <button type="button"
                                                    class="sitycloud-btn sitycloud-btn-outline"
                                                    onclick="window.open('<?php echo esc_url( $home_url ); ?>','_blank');">
                                                فتح الموقع
                                            </button>
                                        <?php endif; ?>
                                    </div>

                                </div>
                            </div>
                        </div>
                        <?php
                    endforeach;
                endif;
                ?>
            </div>
        </div>
    </div>
    <?php
}