'use strict';

/**
 * This file handles all the ajax requests for the core dashboard.
 * This also includes generating and charts and populating the data into a view.
 * 
 */

jQuery(document).ready(function ($) {

	/**
	 * Handles the ajax request for the dashboard stats.
	 *
	 * @param {string} endpoint The API endpoint.
	 * @param {object} data     The data to send in the request.
	 * 
	 * @returns {object}
	 */
	const fetch_stats = async (endpoint, data = {}) => {

		// start/end dates and date differ.
		if ($('#analytify_date_start').length) {
			data.sd = $('#analytify_date_start').val();
		}
		if ($('#analytify_date_end').length) {
			data.ed = $('#analytify_date_end').val();
		}
		if ($('#analytify_date_diff').length) {
			data.d_diff = $('#analytify_date_diff').val();
		}

		// convert data to url params.
		const params = new URLSearchParams();
		for (const key in data) {
			params.append(key, data[key]);
		}

		const URL = `${analytify_stats_forms.url + endpoint + '/' + analytify_stats_forms.delimiter}${params.toString()}`;
		console.log(URL);

		let request = await fetch(URL, {
			method: "GET",
			headers: {
				"X-WP-Nonce": analytify_stats_forms.nonce,
			},
		});

		return request.json();
	}

	/**
	 * Sets the target element to 'loading' state.
	 * Also clear the element's contents.
	 * 
	 * @param {element} target The target element.
	 */
	const prepare_section = (target) => {
		target.find('.analytify_stats_loading').show();
		target.find('.analytify_status_body .stats-wrapper').html('');
		target.find('.analytify_status_footer').remove();
		target.find('.empty-on-loading').html('');
	}

	/**
	 * Element should not be loading.
	 * 
	 * @param {element} target The target element.
	 */
	const should_not_be_loading = (target) => {
		target.find('.analytify_stats_loading').hide();
	}

	/**
	 * Sets the content for the element.
	 * Removes the 'loading' state.
	 * 
	 * @param {element} target     The target element.
	 * @param {string}  markup     The markup to be set (should be HTML).
	 * @param {string}  footer     The footer markup to be set.
	 * @param {boolean} pagination Whether to show the pagination or not.
	 */
	const set_section = (target, markup, footer = false, pagination = false) => {
		should_not_be_loading(target);
		target.find('.analytify_status_body .stats-wrapper').html(markup);

		if (footer || pagination) {
			let footer_markup = `<div class="analytify_status_footer">
				${footer ? `<span class="analytify_info_stats">${footer}</span>` : ''}
				${pagination ? `<div class="wp_analytify_pagination"></div>` : ''}
			</div>`;
			target.find('.analytify_status_body').after(footer_markup);
		}
	}

	/**
	 * Generates the empty stats message.
	 * 
	 * @param {element} target       The target element.
	 * @param {string}  message      The message to be shown.
	 * @param {object}  table_header The table header, if the message needs to be shown in a table.
	 * @param {string}  table_class  The table class.
	 */
	const stats_message = (target = false, message = false, table_header = false, table_class = '') => {
		let markup = `<div class="analytify-stats-error-msg">
			<div class="wpb-error-box">
				<span class="blk"><span class="line"></span><span class="dot"></span></span>
				<span class="information-txt">${message ? message : analytify_stats_forms.no_stats_message}</span>
			</div>
		</div>`;

		if (table_header) {
			// If the error message is to be shown in a table.
			let thead = '';
			for (const td_key in table_header) {
				const td = table_header[td_key];
				if (td.label) {
					thead += `<th class="${td.th_class ? td.th_class : ``}">${td.label}</th>`;
				}
			}

			markup = `<table class="${table_class}">${'' !== thead ? `<thead><tr>${thead}</tr></thead>` : ''}<tbody><tr><td class="analytify_td_error_msg" colspan="${Object.keys(table_header).length}">${markup}</td></tr></tbody></table>`;
		}

		if (!target) {
			return markup;
		}
		set_section(target, markup, false, false);
	}

	/**
	 * Generates the red message box.
	 * 
	 * @param {element} target       The target element.
	 * @param {string}  message      The message to be shown.
	 * @param {object}  table_header The table header, if the message needs to be shown in a table.
	 * @param {string}  table_class  The table class.
	 */
	const red_stats_message = (target = false, message = false, table_header = false, table_class = '') => {
		let markup = `<div class="analytify-email-promo-contianer">
			<div class="analytify-email-premium-overlay">
				<div class="analytify-email-premium-popup">
					${message.title ? `<h3 class="analytify-promo-popup-heading" style="text-align:left;">${message.title}</h3>` : ``}
					${message.content ? message.content : ``}
				</div>
			</div>
		</div>`;

		if (table_header) {
			// If the error message is to be shown in a table.
			let thead = '';
			for (const td_key in table_header) {
				const td = table_header[td_key];
				if (td.label) {
					thead += `<th class="${td.th_class ? td.th_class : ``}">${td.label}</th>`;
				}
			}

			markup = `<table class="${table_class}">${'' !== thead ? `<thead><tr>${thead}</tr></thead>` : ''}<tbody><tr><td class="analytify_td_error_msg" colspan="${Object.keys(table_header).length}">${markup}</td></tr></tbody></table>`;
		}

		if (!target) {
			return markup;
		}
		set_section(target, markup, false, false);
	}

	/**
	 * Generates table from object.
	 * {headers} is to generate <thead> tag and content.
	 * {stats} is to generate <tbody> tag and content.
	 * 
	 * @param {object} headers       The headers.
	 * @param {object} stats         The stats to be shown.
	 * @param {string} table_classes Table classes.
	 * @param {string} attr          Table attributes.
	 * @returns {string}
	 */
	const generate_stats_table = (headers, stats, table_classes = false, attr = '') => {

		let markup = ``;

		markup += `<table class="${table_classes}" ${attr}>`;

		let thead = '';
		for (const td_key in headers) {
			const td = headers[td_key];
			if (td.label) {
				thead += `<th class="${td.th_class ? td.th_class : ``}">${td.label}</th>`;
			}
		}
		if ('' !== thead) {
			markup += `<thead><tr>${thead}</tr></thead>`;
		}

		markup += `<tbody>`;

		let i = 1;
		for (const row_id in stats) {
			const row = stats[row_id];
			markup += `<tr>`;
			for (const td_key in row) {

				let __label = '';

				if (row[td_key] === null && headers[td_key].type && 'counter' === headers[td_key].type) {
					__label = i;
				} else if (row[td_key].label) {
					__label = row[td_key].label;
				} else if (row[td_key].value) {
					__label = row[td_key].value;
				} else {
					__label = row[td_key];
				}

				let __class = '';
				if (row[td_key] && row[td_key].class) {
					__class = row[td_key].class;
				} else if (headers[td_key] && headers[td_key].td_class) {
					__class = headers[td_key].td_class;
				}

				markup += `<td class="${__class}">${__label}</td>`;
			}
			markup += `</tr>`;
			i++;
		}
		markup += `</tbody>`;

		markup += `</table>`;

		return markup;

	}

	/**
	 * Returns the compare start and end date based on given start and end date.
	 * 
	 * @param {string} __start_date Start date.
	 * @param {string} __end_date   End date.
	 * @param {bool}   formatted    If true, returns formatted date to be used in the GA's dashboard url.
	 * @returns {object}
	 */
	const calc_compare_date = (__start_date, __end_date, formatted = true) => {

		const append_zero = (__num) => {
			return (__num < 10) ? '0' + __num : __num;
		}

		const compare_date = {};

		const start_date = new Date(__start_date);
		const end_date = new Date(__end_date);

		let diff_in_days = end_date.getTime() - start_date.getTime();
		if (diff_in_days === 0) {
			diff_in_days = 1 * 24 * 60 * 60 * 1000;
		}

		const compare_date_start = new Date(start_date.getTime() - diff_in_days);
		const compare_date_end = new Date(end_date.getTime() - diff_in_days);

		compare_date.start = compare_date_start.getFullYear() + '-' + append_zero(compare_date_start.getMonth() + 1) + '-' + append_zero(compare_date_start.getDate());
		compare_date.end = compare_date_end.getFullYear() + '-' + append_zero(compare_date_end.getMonth() + 1) + '-' + append_zero(compare_date_end.getDate());

		if (!formatted) {
			return compare_date;
		}

		return `%26_u.date00%3D${__start_date.replaceAll('-', '')}%26_u.date01%3D${__end_date.replaceAll('-', '')}%26_u.date10%3D${compare_date.start.replaceAll('-', '')}%26_u.date11%3D${compare_date.end.replaceAll('-', '')}`;
	}

	/**
	 * Builds the GA dashboard link for sections.
	 * Attaches the data parameter dynamically.
	 */
	const build_ga_dashboard_link = () => {
		const __start_date = $('#analytify_date_start').val();
		const __end_date = $('#analytify_date_end').val();

		const date_parameter = calc_compare_date(__start_date, __end_date, true);

		$('[data-ga-dashboard-forms-link]').each(function (index, __element) {
			const link = $(__element).attr('data-ga-dashboard-link') + date_parameter;
			$(__element).attr('href', link);
		});
	}

	/**
	* Fetches the data and build the template for all endpoints.
	* 
	*/
	const build_sections = () => {

		// To trigger same-height js script.
		$(window).trigger('resize');

		$('[data-endpoint-forms]').each(function (index, __element) {
			const element = $(__element);
			const type = element.attr('data-endpoint-forms');

			prepare_section(element);

			if ('main-dashboard' === type) {
				// Data is returned by the same api call for these section.
				prepare_section($('.form-section:not([data-endpoint-forms="main-dashboard"])'));
			}

			fetch_stats(type).then((response) => {

				if (response.success) {

					switch (type) {
						case 'main-dashboard':
							{
								const table_classes = 'wp_analytify_paginated analytify_data_tables';

								for (const form_key in response.stats) {

									const section = $(`#section-${form_key}`);
									const headers = response.headers;

									if ('analytify_form_custom' === form_key) {
										delete headers.id;
									}

									if (Object.keys(response.stats[form_key].stats).length > 0) {
										const markup = generate_stats_table(headers, response.stats[form_key].stats, table_classes, 'data-product-per-page="7"');
										set_section(section, markup, response.stats[form_key].info_message, true);
									} else {
										stats_message(section, false, headers, table_classes);
									}
								}
							}
							break;

						default:
							break;
					}

					build_ga_dashboard_link();
					wp_analytify_paginated();
					$(window).trigger('resize');

				} else {
					should_not_be_loading(element);
					stats_message(element, analytify_stats_forms.error_message, response.headers);

					should_not_be_loading($('.form-section:not([data-endpoint-forms="main-dashboard"])'));

				}

			}).catch(function (error) {
				console.log(error);
			});

		});
	}

	build_sections();

	// Call of submission of date form.
	$('form.analytify_form_date').on('submit', function (e) {
		if (analytify_stats_forms.load_via_ajax) {
			e.preventDefault();
			build_sections();
			build_ga_dashboard_link();
		}
	});

});
