<?php
// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	die;
}

class Analytify_Authors_Loader {

	public static $instance;
	public $version;

	public static function get_instance() {
		if ( ! isset( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	public function __construct() {
		$this->version = '6.1.0';
		$this->define_constants();
		$this->hooks();
		$this->may_be_load_addon();
	}

	public function may_be_load_addon() {
		if (
			file_exists( WP_PLUGIN_DIR . '/wp-analytify-pro/wp-analytify-pro.php' ) &&
			function_exists( 'wp_analytify_pro_load' ) &&
			class_exists( 'WP_Analytify_Pro_Base' )
		) {
			// The core plugin class.
			require plugin_dir_path( __FILE__ ) . 'classes/analytify-author-tracking.php';
			require plugin_dir_path( __FILE__ ) . 'classes/class-analytify-author-rest-api.php';

			new Analytify_Addon_Authors();
		} else {
			add_action( 'admin_enqueue_scripts', array( $this, 'wpa_authors_dashboard_notices_scripts' ) );
		}
	}

	public function define_constants() {
		// Currently plugin version.
		define( 'ANALYTIFY_PRO_AUTHORS_DASHBOARD_VERSION', '6.1.0' );

		// The URL of the plugin directory.
		define( 'ANALYTIFY_PRO_AUTHORS_ROOT_PATH', __DIR__ );

		// The minimum version of Analytify.
		define( 'ANALYTIFY_PRO_AUTHORS_DASHBOARD_MIN_V_ANALYTIFY', '4.0.0' );

		// The minimum version of Analytify Pro.
		define( 'ANALYTIFY_PRO_AUTHORS_DASHBOARD_MIN_V_ANALYTIFY_PRO', '4.0.0' );
	}

	public function hooks() {
		add_action( 'admin_notices', array( $this, 'wpa_authors_dashboard_install_analytify_pro' ) );
		add_action( 'admin_notices', array( $this, 'wpa_authors_dashboard_active_analytify_pro' ) );
		add_action( 'admin_notices', array( $this, 'wpa_authors_dashboard_notice_version_analytify' ) );
		add_action( 'admin_notices', array( $this, 'wpa_authors_dashboard_notice_version_analytify_pro' ) );
		add_action( 'init', array( $this, 'analytify_authors_load_textdomain' ) );
	}

	/**
	 * Check If Analytify Pro is installed.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function wpa_authors_dashboard_install_analytify_pro() {
		if ( ! file_exists( WP_PLUGIN_DIR . '/wp-analytify-pro/wp-analytify-pro.php' ) ) {
			$message = '<strong>' . esc_html__( 'Attention:', 'wp-analytify-authors' ) . '</strong> ' . esc_html__( 'Please Install Analytify Pro to use Analytify Authors Dashboard addon.', 'wp-analytify-authors' );
			$this->wpa_authors_dashboard_notice_markup( $message, 'wp-analytify-danger' );
		}
	}

	/**
	 * Admin notices for the Analytify version.
	 *
	 * @return void
	 */
	public function wpa_authors_dashboard_notice_version_analytify() {
		// Load the main plugin file.
		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$plugins = get_plugins();
		// Check for the main plugin version.
		if ( ! isset( $plugins['wp-analytify/wp-analytify.php'] ) || version_compare( $plugins['wp-analytify/wp-analytify.php']['Version'], ANALYTIFY_PRO_AUTHORS_DASHBOARD_MIN_V_ANALYTIFY, '<' ) ) {
			// translators: %s: analytify min supported version.
			$message = '<strong>' . esc_html__( 'Attention:', 'wp-analytify-authors' ) . '</strong> ' . sprintf( esc_html__( 'The minimum version of Analytify required for Analytify Authors Dashboard addon is %s. You need to update Analytify Core (Free).', 'wp-analytify-authors' ), ANALYTIFY_PRO_AUTHORS_DASHBOARD_MIN_V_ANALYTIFY );
			$this->wpa_authors_dashboard_notice_markup( $message, 'wp-analytify-danger' );
		}
	}

	/**
	 * Admin notices for the Analytify Pro version.
	 *
	 * @return void
	 */
	public function wpa_authors_dashboard_notice_version_analytify_pro() {
		// Load the main plugin file.
		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$plugins = get_plugins();
		// Check for the pro version.
		if ( ! isset( $plugins['wp-analytify-pro/wp-analytify-pro.php'] ) || version_compare( $plugins['wp-analytify-pro/wp-analytify-pro.php']['Version'], ANALYTIFY_PRO_AUTHORS_DASHBOARD_MIN_V_ANALYTIFY_PRO, '<' ) ) {
			// translators: %s: analytify min supported version.
			$message = '<strong>' . esc_html__( 'Attention:', 'wp-analytify-authors' ) . '</strong> ' . sprintf( esc_html__( 'The minimum version of Analytify Pro required for Analytify Authors Dashboard addon is %s. You need to update Analytify Pro.', 'wp-analytify-authors' ), ANALYTIFY_PRO_AUTHORS_DASHBOARD_MIN_V_ANALYTIFY );
			$this->wpa_authors_dashboard_notice_markup( $message, 'wp-analytify-danger' );
		}
	}

	/**
	 * Check if Analytify Pro is activated.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function wpa_authors_dashboard_active_analytify_pro() {
		if ( ! class_exists( 'WP_Analytify_Pro_Base' ) ) {
			$message = '<strong>' . esc_html__( 'Attention:', 'wp-analytify-authors' ) . '</strong> ' . esc_html__( 'Please activate Analytify Pro to use Analytify Authors Dashboard addon.', 'wp-analytify-authors' );
			$this->wpa_authors_dashboard_notice_markup( $message, 'wp-analytify-danger' );
		}
	}

	/**
	 * Dashboard notices markup.
	 *
	 * @param string $message Message to display.
	 * @param string $class   Class to add to the notice.
	 *
	 * @return void
	 */
	function wpa_authors_dashboard_notice_markup( $message, $class = 'wp-analytify-success' ) {
		$notice_id = 'analytify_notice_' . md5( $message );
		if ( ! get_option( $notice_id ) ) {
			echo '<div class="wp-analytify-notification '. $class .'" id="'. $notice_id .'">
					<a class="wp-analytify-notice-dismiss" href="#" aria-label="Dismiss the notice" onclick="dismissAnalytifyNotice(\''. $notice_id .'\'); return false;">&times;</a>
					<div class="wp-analytify-notice-logo">
						<img src="' . plugins_url( 'assets/img/logo.svg', __FILE__ ) . '" alt="notice">
					</div>
					<div class="wp-analytify-notice-discription">
						<p>' . $message .'</p>
					</div>
				</div>';
		}
	}
    
    public function wp_analytify_authors_notice_script() {
        ?>
        <script type="text/javascript">
            function dismissAnalytifyNotice(noticeId) {
                document.getElementById(noticeId).style.display = 'none';
                localStorage.setItem('analytify_stats_refreshed_dismissed', '1');
                var ajaxurl = '<?php echo admin_url('admin-ajax.php'); ?>';
                var data = {
                    'action': 'dismiss_analytify_notice',
                    'notice_id': noticeId
                };
                jQuery.post(ajaxurl, data, function(response) {
                    console.log(response);
                });
            }
        </script>
        <?php
    }
    
    public function dismiss_authors_campaigns_notice() {
        if ( isset( $_POST['notice_id'] ) ) {
            update_option( sanitize_text_field( $_POST['notice_id'] ), true );
            wp_send_json_success();
        } else {
            wp_send_json_error();
        }
    }

	/**
	 * Admin notices scripts.
	 *
	 * @return void
	 */
	public function wpa_authors_dashboard_notices_scripts() {
		wp_enqueue_style( 'analytify_authors_dashboard_notices', plugins_url( 'assets/css/notices.css', __FILE__ ), ANALYTIFY_PRO_AUTHORS_DASHBOARD_VERSION );
	}

	/**
	 * Load TextDoamin
	 *
	 * @since 1.0
	 */
	public function analytify_authors_load_textdomain() {
		$plugin_dir = basename( __DIR__ );
		load_plugin_textdomain( 'wp-analytify-authors', false, $plugin_dir . '/languages/' );
	}
}

Analytify_Authors_Loader::get_instance();