/**
 * Handles the document ready event and initializes various functionalities.
 * 
 * @param {Object} $ - jQuery object.
 */
jQuery(document).ready(function ($) {

    /**
     * Tracks the state of license registration AJAX requests.
     * @type {boolean}
     */
    var doing_license_registration_ajax = false;

    /**
     * Constructs the admin URL and spinner URL based on the device pixel ratio.
     * @type {string}
     */
    var admin_url = ajaxurl.replace('/admin-ajax.php', ''),
        spinner_url = admin_url + '/images/spinner';

    if (2 < window.devicePixelRatio) {
        spinner_url += '-2x';
    }
    spinner_url += '.gif';

    /**
     * HTML for the AJAX spinner.
     * @type {string}
     */
    var ajax_spinner = '<img src="' + spinner_url + '" alt="" class="ajax-spinner general-spinner" />';

    /**
     * Handles the click event for activating the campaigns license.
     * 
     * @param {Event} e - The click event object.
     */
    $(document).on('click', "#analytify_campaigns_license_activate", function (e) {
        e.preventDefault();

        if (doing_license_registration_ajax) {
            return;
        }

        $('#campaigns-license-status').removeClass('notification-message error-notice');

        /**
         * Retrieves and trims the license key input value.
         * @type {string}
         */
        var license_key = $.trim($("#analytify_campaigns_license_key").val());

        if ('' === license_key) {
            $('#campaigns-license-status').addClass('notification-message error-notice');
            $('#campaigns-license-status').html(wpanalytify_strings.enter_license_key);
            return;
        }

        $('#campaigns-license-status').empty().removeClass('success-notice');
        doing_license_registration_ajax = true;
        $('#analytify_campaigns_license_activate').after('<img src="' + spinner_url + '" alt="" class="register-license-ajax-spinner general-spinner" />');

        /**
         * Sends an AJAX request to activate the license.
         */
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            dataType: 'JSON',
            cache: false,
            data: {
                action: 'wpanalytify_campaigns_activate_license',
                campaigns_license_key: license_key,
                nonce: wpanalytify_data.nonces.activate_license,
                context: 'license'
            },
            error: function (jqXHR, textStatus, errorThrown) {
                doing_license_registration_ajax = false;
                $('.register-license-ajax-spinner').remove();
                $('#campaigns-license-status').html(wpanalytify_strings.register_license_problem);
            },
            success: function (data) {
                doing_license_registration_ajax = false;
                $('.register-license-ajax-spinner').remove();

                if ('undefined' !== typeof data.error) {
                    $('#campaigns-license-status').addClass('notification-message error-notice');
                    $('#campaigns-license-status').html(data.error);
                } else if (data == '0') {
                    $('#campaigns-license-status').addClass('notification-message error-notice');
                    $('#campaigns-license-status').html(wpanalytify_strings.register_license_problem);
                } else {
                    $('#campaigns-license-status').html(wpanalytify_strings.license_registered).delay(5000).fadeOut(1000);
                    $('#campaigns-license-status').addClass('notification-message success-notice');
                    $('#analytify_campaigns_license_key, #analytify_campaigns_license_activate').remove();
                    $('.campaigns-license-row').prepend(data.masked_license);
                }
            }
        });
    });

    /**
     * Initializes pagination for campaign tables.
     */
    wp_analytify_campaign_paginated = function () {
        $('table.wp_analytify_campaign_paginated').each(function () {
            var $table = $(this);
            var $rowsPerPageButtons = $('<div class="wp_analytify_pager"></div>');
            var rowsOptions = [10, 25, 50];
            var totalRows = $table.find('tbody tr').length;
            
            // Don't show pagination if there are no rows
            if (totalRows === 0) {
                return;
            }

            /**
             * Filters the row options to show only those less than the total rows.
             * @type {Array<number>}
             */
            var shownOptions = rowsOptions.filter(function (opt) { 
                return opt < totalRows; 
            });

            /**
             * Finds the next highest option to always show.
             * @type {number|undefined}
             */
            var nextHighest = rowsOptions.find(function (opt) { 
                return opt >= totalRows; 
            });
            if (nextHighest && !shownOptions.includes(nextHighest)) {
                shownOptions.push(nextHighest);
            }
            if (shownOptions.length === 0) shownOptions = [rowsOptions[0]];

            /**
             * Default number of rows per page.
             * @type {number}
             */
            var numPerPage = shownOptions[0];

            /**
             * Adds a button for each rows-per-page option.
             * 
             * @param {number} num - The number of rows to display per page.
             * @param {number} idx - The index of the current option in the shownOptions array.
             */
            shownOptions.forEach(function (num, idx) {
                var $btn = $('<span type="button" class="rows-btn page-number" data-rows="' + num + '">' + num + '</span>');
                if (idx === 0) $btn.addClass('clickable wp_analytify_active');
                $rowsPerPageButtons.append($btn);
            });

            // Only show the row selection buttons in the pagination wrapper
            $(this).next('.wp_analytify_campaign_pagination_wraper').html($rowsPerPageButtons);

            $rowsPerPageButtons.on('click', '.rows-btn', function () {
                numPerPage = parseInt($(this).data('rows'), 10);
                $rowsPerPageButtons.find('.rows-btn').removeClass('clickable wp_analytify_active');
                $(this).addClass('clickable wp_analytify_active');
                $table.trigger('repaginate');
            });

            $table.bind('repaginate', function () {
                $table.find('tbody tr').hide();
                var $filteredRows = $table.find('tbody tr');
                $filteredRows.slice(0, numPerPage).show();
            });

            $table.trigger('repaginate');
        });
    };

    /**
     * Handles the click event for viewing campaign details.
     * 
     * @param {Event} event - The click event object.
     */
    $(document).on('click', '.analytify-view-detail-campaings', function (event) {
        event.preventDefault();

        // Don't send the request if stats are already loaded.
        if ('disabled' == $(this).attr('disabled')) {
            return;
        }

        $(this).attr('disabled', 'disabled');
        const campaign_name = $(this).data('source');
        const parent_container = $(this).closest('.pa_campaigns_statistics');

        /**
         * Sends an AJAX request to fetch campaign details.
         */
        $.ajax({
            url: JSON.parse(analytify_campaigns.resturl),
            data: {
                'cn': campaign_name,
                'sd': $("#analytify_date_start").val(),
                'ed': $("#analytify_date_end").val(),
            },
            beforeSend: function (xhr) {
                xhr.setRequestHeader('X-WP-Nonce', analytify_campaigns.restnonce);
                parent_container.addClass("campaigns_statistics_active").find('.analytify-detail-campaings-container').addClass("stats_loading");
            },
        })
            .done(function (data) {
                parent_container.find('.analytify-detail-campaings-container').removeClass("stats_loading").html(data.body);
                wp_analytify_campaign_paginated();
                enableAnalytifyTableSorting();
            });

    });

    // Add sorting to wp_analytify_campaign_paginated tables
    enableAnalytifyTableSorting();
});
